/*
 * lptlib - routines to access parallel port
 */
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/sysmacros.h>

#ifdef HAVE_PPDEV_H
#include <linux/parport.h>
#include <sys/ioctl.h>
#include <linux/ppdev.h>
#endif

#ifdef HAVE_PPUSER_H
#include <linux/parport.h>
#include <sys/ioctl.h>
#include <linux/ppuser.h>
#endif

#include "lptlib.h"

#if defined(HAVE_PPUSER_H) || defined(HAVE_PPDEV_H)

int opendev(char *dev)
{
	int fd;
	struct stat s;

	fd = open(dev, O_RDWR|O_NONBLOCK);
	if(fd < 0) {
		return -1;
	}

	
	if(ioctl(fd, PPCLAIM, 0) != 0) {
#ifdef DEBUG
		fprintf(stderr, "claim %s: %s\n", dev, strerror(errno));
#endif
		close(fd);
		return -1;
	}
	return fd;
}

void
closedev(int fd)
{
	close(fd);
}

int rd_data(int base)
{
	int d;
	unsigned char byte;

	ioctl(base, PPRDATA, &byte);
	d = byte;

#ifdef DEBUG
	fprintf(stderr, "rd 0x%02x\n", (int)d);
#endif
	return d;
}

void wr_data(int base, int val)
{
	unsigned char byte;
#ifdef DEBUG
	fprintf(stderr, "wr_data fd=%d val=0x%02x\n", base, val);
#endif

	byte = val & 0xff;
	ioctl(base, PPWDATA, &byte);
}

int rd_ctrl(int base)
{
	unsigned char byte;
	ioctl(base, PPRCONTROL, &byte);
	return byte;
}

void wr_ctrl(int base, int val)
{
	unsigned char byte;
	byte = val & 0xff;
	ioctl(base, PPWCONTROL, &byte);
}

int rd_status(int base)
{
	unsigned char byte;
	ioctl(base, PPRSTATUS, &byte);
	return byte;
}


#else

/* A whole seperate implementation that uses ioperm and inb/outb 
 * This only works for legacy ISA-bus ports on intel 386 systems.
 */

#include <asm/io.h>


static int lp_addrs[] = {
  0x3bc,
  0x378,
  0x278,
};
#define LPNO (sizeof(lp_addrs)/sizeof(int))
#define LPSIZE 3

#define OFF_DATA 0
#define OFF_STAT 1
#define OFF_CTRL 2

int opendev(char *dev)
{
	int base, fd;
	struct stat s;

	/* 
	 * figure out a base address and do an ioperm() call.
	 * We use the minor device number of the device name to index a table
	 * that hopefully is like the one in the kernel.  It is for 2.0
	 * kernels, but with 2.2, where parallel device minor numbers are
	 * allocated in the order they're found, this will seldom match up.
	 */
	if(stat(dev, &s) < 0) {
		perror(dev);
		return -1;
	}
	if(!S_ISCHR(s.st_mode)) {
		fprintf(stderr, "%s: not a character device\n", dev);
		return -1;
	}
	if(minor(s.st_rdev) >= LPNO) {
		fprintf(stderr, "no address for minor device %d\n", minor(s.st_rdev));
		return -1;
	} else {
		base = lp_addrs[minor(s.st_rdev)];
	}
	
	if (ioperm(base, LPSIZE, 1) < 0) {
		fprintf(stderr, "%s: ioperm:%s ", dev, strerror(errno));
		return -1;
	}
#ifdef DEBUG
	fprintf(stderr, "ioperm succeded; base=0x%x\n", base);
#endif
	return base;
}

int rd_data(int base)
{
	int d;
	unsigned char byte;

	d = inb(base+OFF_DATA);
#ifdef DEBUG
	fprintf(stderr, "rd 0x%02x\n", (int)d);
#endif
	return d;
}

void wr_data(int base, int val)
{
	outb(val, base+OFF_DATA);
}

int rd_ctrl(int base)
{
	return inb(base+OFF_CTRL);
}

void wr_ctrl(int base, int val)
{
	outb(val, base+OFF_CTRL);
}

int rd_status(int base)
{
	return inb(base+OFF_STAT);
}



#endif
