///////////////////////////////////////////////////////////////////////////
//
// in2cmd.C -- convert .in file into a spice command file
//              Writes output to stdout
//
// Renamed and forked off from in2cazm.C Jan 7, 1999 by Steve Tell.
// Translated from the awk 6/23/94 by Steve Molnar
//
// $Log: in2cmd.C,v $
// Revision 1.16  2003/09/03 19:53:27  cvs
// bump revision number to 20030903
// improve comment output in incat
// add support for M= parameter to hspice-monte
// stuff for compilation on latest gcc
//
// Revision 1.15  2000/12/07 02:28:05  cvs
// Add a little hack to insert literal strings into command file
//
// Revision 1.14  1999/11/01 22:24:07  tell
// rename emit_plot_* to emit_probe.  Create emit_probe_string by splitting up
// emit_probe_sig.  Support for probe= attribute.
//
// Revision 1.13  1999/08/24 22:37:34  tell
// Add -C option to specify clock signal names and waveform template names
//
// Revision 1.12  1999/02/26 03:13:00  tell
// minor cleanup:
// change name of default output to "spice"
// use .probe for outputs. remove .options post=2 probe;
// spicepp templates assumed to handle spice-dialect-specific stuff
//
// Revision 1.11  1999/02/18 00:55:11  tell
// Transform in2cazm into in2cmd
// Can now write either hspice+spicepp or cazm formats
//
// Revision 1.10  1998/09/24 18:34:24  tell
// minor fixes to sync up with changes in inparse.h
//
//
//////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include "inparse.h"

static char **InVecs;
static int *InVecEdge;
static int WarnedSig[MAX_SIGS];

#define DEFAULT_TSTEP 0.1		/* timestep for transient statement */
static double tstep = DEFAULT_TSTEP; /* in nanoseconds */

#define DEFAULT_CLOCK_NAME "Clk"
#define DEFAULT_CLOCK_TEMPLATE "Clock"

struct clkSig {
	char *name;
	char *clktemplate;
};

#define MAX_CLOCKS 15
struct clkSig Clocks[MAX_CLOCKS] = {
{ DEFAULT_CLOCK_NAME, DEFAULT_CLOCK_TEMPLATE },
{ NULL,NULL }
};
int nClocks = 0;

/* output formats supported.
 *   CAZM output assumed to be preprocessed through cazmpp
 *   SPICE format assumed to be preprocessed through spicepp
 */
#define FMT_SPICE	0
#define FMT_CAZM	1

static int OutFormat = FMT_SPICE;

void emit_probe_start();
void emit_probe_signal(char *sig);
void emit_probe_string(char *sig);
void emit_probe_end();
void emit_transient(int ticks, double step);
void emit_dc_voltage(char *name, char *voltage);
void emit_bit_start(char *name);
void emit_bit_end(double fanout, double delay, char edge, char *wavename);

///////////////////////////////////////////////////////////////////////////
// Print out usage message
//////////////////////////////////////////////////////////////////////////
void usage()
{
	fprintf(stderr, "Usage: in2cmd [options] <infile>\n");
	fprintf(stderr, "options:\n");
	fprintf(stderr, "  -P selects powerup initialization\n");
	fprintf(stderr, "  -N omits inputs from spice plot output\n");
	fprintf(stderr, "  -c <format>  set output dialect;\n");
	fprintf(stderr, "                available formats are \"cazm\" and \"spice\".\n");
	fprintf(stderr, "  -t <step>  sets timestep for transient command; default %0.2f ns\n", DEFAULT_TSTEP);
	fprintf(stderr, "  -C <clocksig>,<template>  Use named clock signal and template instead of\n");

	fprintf(stderr, "  -o <literal-output-string>  Insert string into cmd file\n");
 
	fprintf(stderr, "        or in addition to default %s,%s\n", DEFAULT_CLOCK_NAME, DEFAULT_CLOCK_TEMPLATE);
}

char edge_chars[] = "?FRB"; /* order must be same as defines in inparse.h */

///////////////////////////////////////////////////////////////////////////
// Main routine
//////////////////////////////////////////////////////////////////////////

int main(int argc, char **argv)
{
	char outvec[MAX_SIGS];
	char invec[MAX_SIGS];
	char line  [MAX_LINE_LEN];
	char  *fname;
	int    powerup = 0;
	int    plotins = 1;
	int    inflag, outflag;
	int    i, j, c;
	int    en_vecs;  // estimated expected number of vectors
	int	nvec = 0; // counts actual vectors as read
	int	debug = 0;
	extern char *optarg;
	extern int optind;
	int first_vec = 1;
	char *cp;
	
	// Parse command line
	while((c = getopt(argc, argv, "C:NPc:o:t:x")) != -1) {
		switch(c) {
		case 'C':
			if((cp = strchr(optarg, ',')) != NULL && cp[1]) {
				*cp = 0;
				Clocks[nClocks].name = optarg;
				Clocks[nClocks].clktemplate = cp+1;
				nClocks++;
			}
			break;
		case 'N':
			plotins = 0;
			break;
		case 'P':
			powerup = 1;
			break;
		case 'c':
			if(strcmp(optarg, "spice")==0)
				OutFormat = FMT_SPICE;
			else if(strcmp(optarg, "cazm")==0)
				OutFormat = FMT_CAZM;
			else {
				usage();
				exit(1);
			}
			break;

		case 'o':
			printf("%s\n", optarg);
			break;

		case 't':
			tstep = atof(optarg);
			break;
		case 'x':
			debug = 1;
			break;
		default :
			usage();
			exit(1);
		}
	}
	if(optind + 1 != argc) {
		usage();
		exit(1);
	} else
		fname = argv[optind];
	

	// Parse infile header
	parse_infile_header(fname);

	// Print out preamble part of .cmd file
	emit_dc_voltage("Vdd", "vsupply");

	for(i = 0; Clocks[i].name; i++) {
		switch(OutFormat) {
		case FMT_SPICE:
			printf("@clock %s GND %s\n",
			       Clocks[i].name, Clocks[i].clktemplate);
			break;
		case FMT_CAZM:
			printf("wfm %s\n", Clocks[i].name);
			break;
		}
	}
	printf("\n");
	
	// Print out static signals
	for (i = 0; i < StatNumSigs; i++) {
		if (StatSigs[i].val == 0)
			emit_dc_voltage(StatSigs[i].name, "0");
		else if (StatSigs[i].val == 1)
			emit_dc_voltage(StatSigs[i].name, "vsupply");
		else {
			fprintf(stderr, "Unexpected value %d for static signal value %s\n", StatSigs[i].val, StatSigs[i].name);
			exit(2);
		}
	}
	if(StatNumSigs)
		printf("\n");

	// Read all of the input bit vectors into memory.
	// together with the writing of the wave statements,
	// this corner-turns the input bits.
	int edgetype = 0;
	int expect_edge;
	if(InfileActiveEdge == EDGE_RISE)
		expect_edge = EDGE_RISE;
	else
		expect_edge = EDGE_FALL;

	while (parse_infile_vector(&inflag, invec, &outflag, outvec,
				   line, &edgetype, NULL) != 1) {

		if(inflag==0)
			continue;

		if(first_vec) {
			// allocate space for input bits.  Have to wait
			// until we've seen the first vector so we can
			// guess how many vectors accurately.
			en_vecs = get_infile_num_vectors(outflag);
			if(debug) {
				fprintf(stderr, "in2cmd: estimating there will be %d vectors\n", en_vecs);
			}
			if((InVecs = (char **)malloc(en_vecs*sizeof(char *))) == NULL) {
				fprintf(stderr, "out of memory allocating InVecs\n");
			}
			for(i = 0; i < en_vecs; i++) {
				if((InVecs[i] = (char *)malloc((InNumSigs+1)*sizeof(char))) == NULL) {
					fprintf(stderr, "out of memory allocating InVecs[%d]\n", i);
					exit(1);
				}
			}
			if((InVecEdge = (int *)malloc(en_vecs*sizeof(int))) == NULL) {
				fprintf(stderr, "out of memory allocating InVecEdge\n");
			}
			first_vec = 0;
		}
		memcpy(InVecs[nvec], invec, InNumSigs);

		/* Check vector clock edge types.
		* If both types of signals are present in the .in file,
		* R and F vectors must alternate in vector file.
		* Else, there must be only the correct type.
		*/

		if(InfileActiveEdge == EDGE_BOTH) {
			if(edgetype == EDGE_DEFAULT || edgetype != expect_edge) {
				infile_error(1, "Vector %d: got edge type '%c', expected '%c'\n",
					      nvec, edge_chars[edgetype],
					      edge_chars[expect_edge]);
			}
			if(expect_edge == EDGE_RISE)
				expect_edge = EDGE_FALL;
			else
				expect_edge = EDGE_RISE;


		} else if(edgetype != EDGE_DEFAULT && InfileActiveEdge != edgetype) {
			infile_error(1, "Vector %d: edge type '%c', expected '%c'\n",
				      nvec, edge_chars[edgetype], edge_chars[InfileActiveEdge] );
		} else {
			edgetype = InfileActiveEdge;
		}
		InVecEdge[nvec] = edgetype;

		nvec++;
		if (nvec >= en_vecs) {
			fprintf(stderr, "Estimated number of test vectors (%d) exceeded;\n", en_vecs);
			fprintf(stderr, "  Heuristic in get_infile_num_vectors() must have failed.\n");
			exit(2);
		}

	}

	// Print out the "wave" statements.
  
	for (i = 0; i < InNumSigs; i++) {
		emit_bit_start(InSigs[i].name);
		for (c = 0; c < nvec; c++) {
			if((InVecEdge[c] & InSigs[i].edgestyle) == 0) {
				if(InVecs[c][i] != '-' && !WarnedSig[i]) {
					fprintf(stderr, "Warning: '%c' on input signal %s at inactive edge ignored; '-' recommended\n", InVecs[c][i], InSigs[i].name);
					WarnedSig[i] = 1;
				}
			} else
				putchar(InVecs[c][i]);
		}
		if(InSigs[i].edgestyle != EDGE_FALL || InSigs[i].wavename) {
			emit_bit_end(InSigs[i].fanout, InSigs[i].delay,
				     edge_chars[InSigs[i].edgestyle],
				     InSigs[i].wavename);
		} else {
			emit_bit_end(InSigs[i].fanout, InSigs[i].delay, 0, NULL);
		}
	}
	printf("\n");

	// print out the transient control statement
	// if active on both edges, are twice as many vectors as clock cycles.
	if(InfileActiveEdge == EDGE_BOTH)
		emit_transient(1+nvec/2, tstep);
	else
		emit_transient(1+nvec, tstep);

	if(powerup && OutFormat == FMT_CAZM)
		printf(" powerup");
	printf("\n");

	// print out the plot control statement
	emit_probe_start();
	emit_probe_signal(Clocks[0].name);
	if(plotins)
		for (i = 0; i < ISigs.nSigs; i++)
			emit_probe_signal(ISigs.Sigs[i].name);

	// output signals
	for (i = 0; i < OSigs.nFields; i++) {
		if(OSigs.Fields[i].probestr) {
			emit_probe_string(OSigs.Fields[i].probestr);
		} else {
			for (j = 0; j < OSigs.Fields[i].nsigs; j++)
				emit_probe_signal(OSigs.Sigs[OSigs.Fields[i].firstsig+j].name);
		}
	}

	// monitored signals
	for (i = 0; i < MonSigs.nFields; i++) {
		if(MonSigs.Fields[i].probestr) {
			emit_probe_string(MonSigs.Fields[i].probestr);
		} else {
			for (j = 0; j < MonSigs.Fields[i].nsigs; j++)
				emit_probe_signal(MonSigs.Sigs[MonSigs.Fields[i].firstsig+j].name);
		}
	}

	// extra clocks go at the end, where they won't bother spice2out
	for(i = 1; Clocks[i].name; i++) {
		emit_probe_signal(Clocks[i].name);
	}
	emit_probe_end();

	
	switch(OutFormat) {
	case FMT_SPICE:
		/* .end done in spicepp template now */
		/* printf(".end\n"); */
		break;
	}
	exit(0);
}

static int plot_line_len;
static int plot_line_nitems;

void
emit_probe_start()
{
	switch(OutFormat) {
	case FMT_SPICE:
		printf(".probe tran");
		plot_line_len = 11;
		break;
	case FMT_CAZM:
		printf("plot {");
		break;
	}
	plot_line_nitems = 0;
}

void
emit_probe_signal(char *sig)
{
	char *s;

	switch(OutFormat) {
	case FMT_SPICE:
		s = (char *)malloc(strlen(sig)+5);
		sprintf(s, "v(%s)", sig);
		break;
	case FMT_CAZM:
		s = sig;
		break;
	}
	emit_probe_string(s);
	if(s != sig)
		free(s);
}

void
emit_probe_string(char *s)
{
	int item_len;
	switch(OutFormat) {
	case FMT_SPICE:
		if(plot_line_nitems > 30) {
			printf("\n.probe tran");
			plot_line_len = 11;
			plot_line_nitems = 0;
		}
		item_len = strlen(s) + 1;
		if(plot_line_len + item_len > 78) {
			printf("\n+");
			plot_line_len = 1;
		}
		printf(" %s", s);
		plot_line_len += item_len;
		break;
	case FMT_CAZM:
		printf(" %s", s);
		break;
	}
	plot_line_nitems++;
}

void
emit_probe_end()
{
	switch(OutFormat) {
	case FMT_SPICE:
		printf("\n");
		break;
	case FMT_CAZM:
		printf("}\n");
		break;
	}
}

void
emit_transient(int ticks, double step)
{
	switch(OutFormat) {
	case FMT_SPICE:
		printf("@trans %.4fn %dclk\n", step, ticks);
		break;
	case FMT_CAZM:
		printf("transient %dclk %.4fn\n", ticks, step);
		break;
	}
}

void
emit_dc_voltage(char *name, char *voltage)
{
	switch(OutFormat) {
	case FMT_SPICE:
		printf("v%s %s GND %s\n", name, name, voltage);
		break;
	case FMT_CAZM:
		printf("volt %s GND %s\n", name, voltage);
		break;
	}
}

void
emit_bit_start(char *name)
{
	switch(OutFormat) {
	case FMT_SPICE:
		printf("@wavebit %s GND {", name);
		break;
	case FMT_CAZM:
		printf("wave %s\tbit {", name);
		break;
	}
}

void
emit_bit_end(double fanout, double delay, char edge, char *wavename)
{
	printf("} ETFO=%g+%g", fanout, delay);
	if(edge && wavename)
		printf(",%c,%s", edge, wavename);
	putchar('\n');
}
