///////////////////////////////////////////////////////////////////////////
//
// in2ir.C -- convert .in file into irsim command file
//            Writes output to stdout
//
// $Log: in2ir.C,v $
// Revision 1.4  2003/09/03 19:53:27  cvs
// bump revision number to 20030903
// improve comment output in incat
// add support for M= parameter to hspice-monte
// stuff for compilation on latest gcc
//
// Revision 1.3  1999/12/30 19:32:46  tell
// Fixed in2vlogtb bug caused by inparse.C error
// minor analyzer-related fixes in in2ir.C
// Added fdel.v/fdel.in to examples
// Add Sim:W tests to irprep test cases
//
// Revision 1.2  1999/11/01 22:25:16  tell
// keep probe= attributes from breaking irsim runs, but printing GND
// for those signals.
//
// Revision 1.1  1999/11/01 21:27:46  tell
// Initial revision
//
//////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include "inparse.h"

static char *progname = "in2ir";
char edge_chars[] = "?FRB"; /* order must be same as defines in inparse.h */

static int WarnedFld[MAX_FIELDS];

static int inVecNo;
static int viewLogic = 0;
static char *setCmd = "set";
static char *clockName = "Clk";
static char allClkNames[1024];
#define MAX_CLKS 10
static char *extraClkNames[MAX_CLKS];
static int n_extraClks = 0;
static int g_verbose;

// forward declarations
static void process_double_vectors(int extra_step, int flush, int interactive);
static void process_single_vectors(int extra_step, int flush, int interactive);

///////////////////////////////////////////////////////////////////////////
// Print out usage info
//////////////////////////////////////////////////////////////////////////
static void 
usage()
{
	fprintf(stderr, "Usage:  %s [options] <infile>\n", progname);
	fprintf(stderr, "        <infile>     is a doirsim .in file\n");
	fprintf(stderr, "  options:\n");
	fprintf(stderr, "        -a           Set up for irsim graphical analyzer window\n");
	fprintf(stderr, "        -c name      Name of clock net.  Default \"Clk\".\n");
	fprintf(stderr, "        -f n         flush every n cycles.\n");
	fprintf(stderr, "        -i           set up for interactive irsim run; bail out at first break comment.\n");
	fprintf(stderr, "        -s           Run extra simulation step after asserting inputs.\n");
	fprintf(stderr, "        -w           Generate variant cmd file format for viewlogic's viewsim\n");

	exit(3);
}

///////////////////////////////////////////////////////////////////////////
// write an input vector as irsim set statements
// inputs:
//	invec - character array of [01X-] with
//	edge -  EDGE_RISE or EDGE_FALL; the clock edge this vector 
//		is associated with. Compared with input signal active edges 
//		to determine which signals to set.
//	exact - if 1, set only signals that match _edge_ exactly
//		if 0, set double-edge signals always.
// returns:
//	number of set statements written out
//////////////////////////////////////////////////////////////////////////

static int
emit_irsim_vector(char *invec, int edge, int single)
{
	int i;
	int j;
	int n = 0;
	char vec[MAX_FIELD_SIGS];

	for (i = 0; i < InNumFields; i++) {
		strncpy(vec, &invec[InFields[i].firstsig], InFields[i].nsigs);
		vec[InFields[i].nsigs] = '\0';
		for(j = 0; j < InFields[i].nsigs; j++) {
			switch(vec[j]) { // Convert conventional to irsim
			case 'X':
			case 'x':
				if(viewLogic)
					vec[j] = 'x';	// Undefined 
				else
					vec[j] = 'u';	// Undefined 
				break;
			case 'Z':
			case 'z':
				if(viewLogic)
					vec[i] = 'z';	// Undriven
				else
					vec[j] = 'x';	// Undriven
				break;
			}
		}
		
		if((InSigs[InFields[i].firstsig].edgestyle == edge) ||
		   (single==0 &&
		       InSigs[InFields[i].firstsig].edgestyle==EDGE_BOTH)) {
			if(strchr(vec, '-') != NULL) {
				fprintf(stderr, "Error in vector; near vector %d: input field %s has value '%s' on active edge\n",
					inVecNo, InFields[i].name, vec);
				exit(2);
			}
			printf("%s __invec_%d %s\n", setCmd, i+1, vec);
			n++;
		} else if((InSigs[InFields[i].firstsig].edgestyle & edge) == 0) {
			if(strspn(vec, "01xu") > 0 && !WarnedFld[i]) {
				fprintf(stderr, "Warning: %s on input field %s at inactive edge ignored; '-' recommended\n",
					vec, InFields[i].name);
				WarnedFld[i]=1;
			}
		}
	}
	return n;
}


///////////////////////////////////////////////////////////////////////////
// Get a vector from infile - wrapper around parse_infile_vector()
// Loops until it gets a vector; prints comments from non-vector lines.
// Handles default edge for backwards compatibility.
//
// inputs: 
//	mostly the same as parse_infile_vector, except:
//		line and comment omitted
//		outvec and outflag omitted
//		inflag omitted
//	expect_edge - edge type expected. aborts if different one found.
// returns:
//	1 on EOF, 0 on valid vector found
//	2 if vector  has 'break' in comment
//	
/////////////////////////////////////////////////////////////////////////
static int
get_infile_vector(char *invec, int *edgep, int expect_edge)
{
	char *comment;
	int outflag;
	int inflag;
	int edge;
	char outvec[MAX_SIGS];
	char line  [MAX_LINE_LEN];

	do {
		if(parse_infile_vector(&inflag, invec, &outflag, 
			       outvec, line, &edge, &comment) == 1)
			return 1;
		if(comment) {
			printf("| %s\n", comment);
			if(strncmp(comment, "break", 5) == 0)
				return 1;
		}
	} while(!inflag);
	inVecNo++;

	if((InfileActiveEdge == EDGE_FALL) && 
	   (edge == EDGE_DEFAULT))
		edge = EDGE_FALL;

	if(edge != expect_edge) {
		infile_error(1, "Vector %d: got edge type '%c', expected '%c'\n",
			     inVecNo, edge_chars[edge],
			     edge_chars[expect_edge]);
	}
	*edgep = edge;

	if(comment && strncmp(comment, "break", 5) == 0)
		return 2;
	else
		return 0;
}

///////////////////////////////////////////////////////////////////////////
// Main routine
//////////////////////////////////////////////////////////////////////////

int
main(int argc, char **argv)
{
	int i, j;
	int c;
	extern char *optarg;
	extern int optind;
	extern int optopt;
	// flags & arguments
	int extra_simstep = 0;
	int flush_threshold = -1;
	int interactive = 0;
	int do_analyzer = 0;
	int got_clockname = 0;

	// parse arguments
	while((c = getopt(argc, argv, "ac:f:iswv")) != -1) {
		switch(c) {
		case 'a':
			do_analyzer = 1;
			break;
		case 'c':
			if(got_clockname == 0) {
				clockName = optarg;
				got_clockname = 1;
			} else {
				extraClkNames[n_extraClks++] = optarg;
			}
			break;
		case 'f':
			flush_threshold = atoi(optarg);
			if(flush_threshold < 1)
				flush_threshold = 1;
			break;
		case 'i':
			interactive = 1;
			break;
		case 's':
			extra_simstep = 1;
			break;
		case 'v':
			g_verbose = 1;
			break;
		case 'w':
			viewLogic = 1;
			setCmd = "assign";
			break;
		default:
			fprintf(stderr, "%s: Unrecognized option: -%c\n", progname, optopt);
			usage();
			break;
		}
	}
	if(optind + 1 != argc) {
		fprintf(stderr, "%s: input file not specified.\n", progname);
		usage();
	}

	InfileActiveEdge = 0;
	// Parse infile header
	parse_infile_header(argv[optind]);

	// force extra step if both edges used.  ir2out can't correctly sample
	// signals without this.  Not a performance issue because
	// double-edge svensson flip-flops don't work in irsim without it.
	if(g_verbose)
		fprintf(stderr, ".in file active edge = %c\n", edge_chars[InfileActiveEdge]);
	if(InfileActiveEdge == EDGE_BOTH)
		extra_simstep = 1;
	
	// Print out static signals
	for (i = 0; i < StatNumSigs; i++) {
		if (StatSigs[i].val == 0) printf("l ");
		else if (StatSigs[i].val == 1) printf("h ");
		else {
			fprintf(stderr, "Unexpected value static signal value %d\n",
				StatSigs[i].val);
			exit(-1);
		}
		printf("%s\n", StatSigs[i].name);
	}

	// Write out the "vector" commands for each group of signals to monitor
	for (i = MonSigs.nFields-1; i >= 0; i--) {
		printf("vector __monvec_%d", i+1);
		for (j = 0; j < MonSigs.Fields[i].nsigs; j++) {
			if(MonSigs.Fields[i].probestr)
				printf(" GND");
			else 
				printf(" %s", MonSigs.Sigs[MonSigs.Fields[i].firstsig+j].name);
		}
		printf("\n");
		if(viewLogic)
			printf("radix bin __monvec_%d\n", i+1);
		if(interactive) {
			printf("w");
			for (j = 0; j < MonSigs.Fields[i].nsigs; j++) {
				if(MonSigs.Fields[i].probestr)
					printf(" GND");
				else 
					printf(" %s", MonSigs.Sigs[MonSigs.Fields[i].firstsig+j].name);
			}
			putchar('\n');
		} else {
			printf("w __monvec_%d\n", i+1);
		}
	}
	// Write out the "vector" commands for each group of outputs
	for (i = OutNumFields-1; i >= 0; i--) {
		printf("vector __outvec_%d", i+1);
		for (j = 0; j < OutFields[i].nsigs; j++) {
			if(OSigs.Fields[i].probestr)
				printf(" GND");
			else
				printf(" %s", OutSigs[OutFields[i].firstsig+j].name);
		}
		printf("\n");
		if(viewLogic)
			printf("radix bin __outvec_%d\n", i+1);
		if(interactive) {
			printf("w");
			for (j = 0; j < OSigs.Fields[i].nsigs; j++) {
				if(OSigs.Fields[i].probestr)
					printf(" GND");
				else
					printf(" %s", OutSigs[OutFields[i].firstsig+j].name);
			}
			putchar('\n');
		} else {
			printf("w __outvec_%d\n", i+1);
		}
	}

	// Write out the "vector" commands for each group of inputs
	for (i = InNumFields-1; i >= 0; i--) {
		printf("vector __invec_%d", i+1);
		for (j = 0; j < InFields[i].nsigs; j++)
			printf(" %s", InSigs[InFields[i].firstsig+j].name);
		printf("\n");
		if(viewLogic)
			printf("radix bin __invec_%d\n", i+1);
		if(interactive) {
			printf("w");
			for (j = 0; j < InFields[i].nsigs; j++)
				printf(" %s", InSigs[InFields[i].firstsig+j].name);
			putchar('\n');
		} else {
			printf("w __invec_%d\n", i+1);
		}
	}
  
	printf("vector __clock__ %s\n", clockName);
	printf("w __clock__\n");
	if(viewLogic)
		printf("radix bin __clock__\n");

	strcpy(allClkNames, clockName);
	for(i = 0; i < n_extraClks; i++) {
		strcat(allClkNames, " ");
		strcat(allClkNames, extraClkNames[i]);
	}

	/* Write out analyzer commands, if requested.
	 *
	 * This needs work, and probably help inside the analyzer itself.
	 * We'd like to give the user the choice of showing a field of signals
	 * as individual lines or as a vector.   If as a vector, they
	 * don't really want the __invec_ or __outvec_ names, but rather something
	 * recognizable.  but what?
	 */
	if(do_analyzer) {
		printf("ana %s\n", clockName);
//		for (i = InNumFields-1; i >= 0; i--) {
		for (i = 0; i < InNumFields; i++) {
			printf("ana");
			for (j = 0; j < InFields[i].nsigs; j++)
				printf(" %s", InSigs[InFields[i].firstsig+j].name);
			putchar('\n');
		}
//		for (i = OutNumFields-1; i >= 0; i--) {
		for (i = 0; i < OutNumFields; i++) {
			printf("ana");
			for (j = 0; j < OutFields[i].nsigs; j++)
				printf(" %s", OutSigs[OutFields[i].firstsig+j].name);
			putchar('\n');
		}
//		for (i = MonSigs.nFields-1; i >= 0; i--) {
		for (i = 0; i < MonSigs.nFields; i++) {
			printf("ana");
			for (j = 0; j < MonSigs.Fields[i].nsigs; j++)
				printf(" %s", MonSigs.Sigs[MonSigs.Fields[i].firstsig+j].name);
			putchar('\n');
		}
	}
	if(viewLogic)
		printf("flush\n");

#ifdef DEBUG
	fprintf(stderr, "InfileActiveEdge=%c\n", edge_chars[InfileActiveEdge]);
#endif
	if(InfileActiveEdge == EDGE_BOTH)
		process_double_vectors(extra_simstep, flush_threshold, interactive);
	else
		process_single_vectors(extra_simstep, flush_threshold, interactive);

	exit(0);
}

////////////////////////////////////////////////////////////////////
// Process vectors for the double-edge case
//	tricky because the vector specification is in terms of
//	what value inputs will have when sampled.  We have to look ahead
//	a half-cycle to assert rising-edge and falling-edge signals so that
//	they have been asserted for almost a full cycle when sampled at their
//	active clock edge.
//
////////////////////////////////////////////////////////////////////

static void process_double_vectors(int extra_step, int flush, int interactive)
{
	int expect_edge;
	static char invec [MAX_SIGS];
	static char invec1 [MAX_SIGS];
	int edge = 0;
	int edge1 = 0;
	int flushc;
	int rc;

	// First vector: must be falling edge, contains initial values
	//   for falling-edge and double-edge signals.
	if(get_infile_vector(invec, &edge, EDGE_FALL) == 1)
		infile_error(1, "EOF reading first vector");
	emit_irsim_vector(invec, EDGE_FALL, 0);

	// second vector: must be rising edge.  contains initial values
	//   for rising-edge signals.
	if(get_infile_vector(invec1, &edge1, EDGE_RISE) == 1)
		infile_error(1, "EOF reading second vector");

	// emit initial values for rising-edge signals, but don't
	// emit second value for double-edge signals yet.
	emit_irsim_vector(invec1, EDGE_RISE, 1);

	// push initial values through circuit.
	printf("h %s\n", allClkNames); // avoids clock=X on this simstep
	printf("s\n");
	
	// clock transition for first falling edge
	printf("l %s\n", allClkNames);
	printf("s\n");
	
	// Do the  initialization cycle
	printf("h %s\n", allClkNames);
	printf("s\n");
	printf("l %s\n", allClkNames);
	printf("s\n");

	expect_edge = EDGE_FALL;
	flushc = 0;

	// Now do the rest of the vectors.
	// This loop runs once per vector, which is once per half cycle.
	// first time through we read the 3rd vector which is falling-edge
	//  and write the falling vector from it and the double-edge
	//  vector from the second vector.
	while ((rc=get_infile_vector(invec, &edge, expect_edge)) != 1) {

//		printf("h DblClk\n");

		emit_irsim_vector(invec, edge, 1);
		emit_irsim_vector(invec1, EDGE_BOTH, 1);
		printf("s\n");

		// clock transition.  direction matches vector following.
		printf("%c %s\n", edge==EDGE_FALL?'h':'l', allClkNames);
//		printf("l DblClk\n");
		printf("s\n");

		if(flush != -1 && ++flushc == flush) {
			printf("flush\n");
			flushc = 0;
		}
		memcpy(invec1, invec, sizeof(invec));
		edge1 = edge;

		if(expect_edge == EDGE_RISE)
			expect_edge = EDGE_FALL;
		else
			expect_edge = EDGE_RISE;

		if(interactive && rc==2) {
			break;
		}
	}

	// One more half-cycle left over due to read-ahead.
	emit_irsim_vector(invec1, EDGE_BOTH, 1);
	printf("s\n");

	// final clock transition.
	printf("%c %s\n", edge1==EDGE_FALL?'l':'h', allClkNames);
	printf("s\n");
	if(interactive && rc==2) {
		printf("print break\n");
	}
}

////////////////////////////////////////////////////////////////////
// Process vectors for the single-edge case
//	Much simpler than the double-edge case, no lookahead required.
//
////////////////////////////////////////////////////////////////////

static void process_single_vectors(int extra_step, int flush, int interactive)
{
	int edgetype = 0;
	int expect_edge;
	static char invec [MAX_SIGS];
	int flushc;
	int rc;

	expect_edge = InfileActiveEdge;

	// Handle initialization cycles.
	// - Run the extra clock cycles to push intial values through
	// - If both edges are active, we have to read ahead to get initial
	//   values for the rising-edge signals out of the second vector.

	if(get_infile_vector(invec, &edgetype, expect_edge) == 1)
		infile_error(1, "EOF reading first vector");

	emit_irsim_vector(invec, edgetype, 0);
	
	// push initial values through
	printf("h %s\n", allClkNames);	// avoids clock=X on this step
	printf("s\n");

	// clock transition for first falling edge
	printf("l %s\n", allClkNames);
	printf("s\n");
	// The initialization cycle
	printf("h %s\n", allClkNames);
	printf("s\n");
	printf("l %s\n", allClkNames);
	printf("s\n");

	// if rising-edge, next vector read in should be 
	// asserted after the second rising edge, so generate the edge.
	// Otherwise, next vector is an 'F' and gets asserted after the
	// second falling edge (the initialization cycle above).
	if(InfileActiveEdge == EDGE_RISE) {
		printf("h %s\n", allClkNames);
		printf("s\n");
	}

	flushc = 0;
	// Now do the rest of the vectors
	// This loop runs once per vector / once per clock cycle
	while ((rc=get_infile_vector(invec, &edgetype, expect_edge)) != 1) {

		emit_irsim_vector(invec, edgetype, 0);
		if(extra_step)
			printf("s\n");

		// clock transition for inactive edge
		printf("%c %s\n", edgetype==EDGE_FALL?'h':'l', allClkNames);
		printf("s\n");

		// active clock edge - the input valus we wrote above
		// will get sampled on this edge by the output processor.
		printf("%c %s\n", edgetype==EDGE_FALL?'l':'h', allClkNames);
		printf("s\n");

		if(flush != -1 && ++flushc == flush) {
			printf("flush\n");
			flushc = 0;
		}

		if(interactive && rc==2) {
			printf("print break\n");
			break;
		}
	}
}
