///////////////////////////////////////////////////////////////////////////
//
// in2vlogtb.C -- Using .in file, create verilog testbench
//            Writes output to stdout
//	May 12, 1998.  Steve Tell.
//
// $Log: in2vlogtb.C,v $
// Revision 1.15  2003/09/03 19:53:27  cvs
// bump revision number to 20030903
// improve comment output in incat
// add support for M= parameter to hspice-monte
// stuff for compilation on latest gcc
//
// Revision 1.14  2002/11/07 23:59:26  cvs
// fix overzealous use of \identifier - broke some real busses
//
// Revision 1.13  2002/11/06 23:19:42  cvs
// overhaul to connect _no_ clock input to dut by default;
// all clocks must be listed with -C options.
// some support for backslashing funny characters in verilog nodenames,
// such as [] for bit-blasted single-wire ports.  Not a general solution though.
//
// Revision 1.12  2001/11/08 23:15:07  cvs
// in2vlogtb, doverilog: Add handling of vport attribute for verilog outputs
// search for verilog simulator to run in $PATH at configure time
// add bus2 example illustrating verilog busses, and some doverilog text to
// examples/Readme
// bump version to 20011108
//
// Revision 1.11  2000/05/18 23:35:28  tell
// Fix stimuli generation startup.  Put out first vector immediatly.
// Do the proper initialization-cycle as per dtools standard, getting
// things correct for double-edge simulations
//
// Revision 1.10  2000/05/15 23:51:31  tell
// Improve some error messages in inparse.C, fix IGNORE_ALL in inparse.h
// Make in2vlogtb.C handle IGNORE_ on inputs: doesn't even drive them.
// (this needs to be propagate to in2ir and in2cmd too)
//
// Revision 1.9  2000/04/19 16:49:41  tell
// bump version to 0.2.21
// make in2vlogtb call $timeformat, and put times in the .vout file
// fix doverilog's handling of multiple --define options
// fix injtag2in to allow # comments in @<directive> statements
//
// Revision 1.8  2000/04/11 18:37:38  tell
// add -a option to enable waveform dumping in SHM or VCD format
// add -p option to set period
//
// Revision 1.7  2000/01/19 21:20:27  tell
// added rudimentary handling of static: signals using verilog force directive.
//
// Revision 1.6  1999/12/30 19:32:46  tell
// Fixed in2vlogtb bug caused by inparse.C error
// minor analyzer-related fixes in in2ir.C
// Added fdel.v/fdel.in to examples
// Add Sim:W tests to irprep test cases
//
// Revision 1.5  1999/11/01 22:24:39  tell
// change parsing callback for vport attribute to use new scheme.
//
// Revision 1.4  1999/06/25 18:21:08  tell
// have the verilog put a header line in the .vout file, for outchk to copy
// into the .logv file.
//
// Revision 1.3  1999/06/25 02:40:19  tell
// added -n option to produce a testbench that includes the input vectors
// in-line instead of using the PLI routine.
//
// Revision 1.2  1998/06/02 18:14:55  tell
// Added support for vectored input ports using vport= attribute
//
// Revision 1.1  1998/05/14 21:45:40  tell
// Initial revision
//
//
//////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include "inparse.h"

static char *progname = "in2vlogtb";
char edge_chars[] = "?FRB"; /* order must be same as defines in inparse.h */

static char *clockName = NULL;
#define MAX_CLKS 10
static char *extraClkNames[MAX_CLKS];
static int n_extraClks = 0;
static int g_verbose;

#define DUMP_VCD	1
#define DUMP_SHM	2
#define DUMP_FSDB	3
#define DUMP_LXT	4

// extra structure added to Field structure
struct fieldAttr {
	char *vport_name;  // verilog port name
};

///////////////////////////////////////////////////////////////////////////
// Print out usage info
//////////////////////////////////////////////////////////////////////////
static void 
usage()
{
	fprintf(stderr, "Usage:  %s [options] <infile>\n", progname);
	fprintf(stderr, "        <infile>     is a doirsim .in file\n");
	fprintf(stderr, "  options:\n");
	fprintf(stderr, "        -a type      generate code for waveform dumping;\n");
	fprintf(stderr, "                     type is vcd or shm\n");
	fprintf(stderr, "        -c name      Name of clock port, if any.\n");
	fprintf(stderr, "        -d name      name of module under test\n");
	fprintf(stderr, "        -m name      name of testbench module\n");
	fprintf(stderr, "        -n           no pli vector-reader, put vectors inline.\n");
	fprintf(stderr, "        -o fname     name of output file generated by testbench\n");
	fprintf(stderr, "        -p period    set clock period\n");
}

///////////////////////////////////////////////////////////////////////////
// Write header
//////////////////////////////////////////////////////////////////////////
void
write_header(char *modulename)
{
	printf("/*\n"); 
	printf(" * verilog testbench created by in2vlogtb\n"); 
	printf(" */\n\n"); 
	printf("module %s;\n\n", modulename); 
}

/*
 * write a wirename for verilog, possibly with a prefix.
 * If the wirename contains special characters, prefix it with a backslash
 * to make verilog interpret it literaly.
 */
void write_wire(char *prefix, char *wirename)
{
	if(strpbrk(wirename, "[],:;{}.~")) {
		putchar('\\');
	}
	printf("%s%s", prefix, wirename);
	putchar(' ');
}

///////////////////////////////////////////////////////////////////////////
// Write declarations for dut input registers
// TODO: implement wire delays in inputs based on ETFO timing spec 
//////////////////////////////////////////////////////////////////////////
void
write_decls(int powernets)
{
	int i, j;

	printf("   reg Clk;\n");
	if(ISigs.nSigs == 1)
		printf("   reg [0:1] inbits;\n");
	else
		printf("   reg [0:%d] inbits;\n", ISigs.nSigs-1);

	if(OSigs.nSigs == 1)
		printf("   reg outbits;\n");
	else
		printf("   reg [0:%d] outbits;\n", OSigs.nSigs-1);
		

	for(i = 0; i < ISigs.nFields; i++) {
		struct fieldAttr *fa;
		fa = (struct fieldAttr *)ISigs.Fields[i].data;

		if((ISigs.Sigs[InFields[i].firstsig].ignore & IGNORE_IRSIM) == 0) {
			if(fa && fa->vport_name) {
				if(ISigs.Fields[i].nsigs > 1) {
					printf("   wire [%d:0] i_%s = inbits[%d:%d];\n",
					       ISigs.Fields[i].nsigs-1,
					       fa->vport_name,
					       ISigs.Fields[i].firstsig, 
					       ISigs.Fields[i].firstsig+ISigs.Fields[i].nsigs - 1);
				} else {
					printf("   wire i_%s = inbits[%d];\n",
					       fa->vport_name,
					       ISigs.Fields[i].firstsig);
				}
			} else {
				for (j = 0; j < ISigs.Fields[i].nsigs; j++) {
					printf("   wire ");
					write_wire("i_", ISigs.Sigs[InFields[i].firstsig+j].name);
					printf("= inbits[%d];\n",
					       ISigs.Fields[i].firstsig+j);
				}
			}
		}
	}
	for(i = 0; i < OSigs.nFields; i++) {
		struct fieldAttr *fa;
		fa = (struct fieldAttr *)OSigs.Fields[i].data;
		if(fa && fa->vport_name) {
			if(OSigs.Fields[i].nsigs > 1) {
				printf("   wire [%d:0] o_%s;\n", 
				       OSigs.Fields[i].nsigs-1, fa->vport_name);
			} else {
				printf("   wire o_%s;\n", fa->vport_name);
			}
		}
	}
	if(powernets) {
		printf("   supply0 GND;\n");
		printf("   supply1 VDD;\n");
	}
	printf("\n");
}

///////////////////////////////////////////////////////////////////////////
// Write instantiation of module under test
//////////////////////////////////////////////////////////////////////////
void
write_minst(char *dut, int powernets)
{
	int i, j;
	printf("   %s dut(\n", dut);
	if(clockName)
		printf("     .%s(Clk),\n", clockName);
	for(i = 0; i < n_extraClks; i++) {
		printf("     .%s(Clk),\n", extraClkNames[i]);
	}
	if(powernets) {
		printf("      .GND(GND),\n");
		printf("      .VDD(VDD),\n");
	}
	for(i = 0; i < ISigs.nFields; i++) {
		struct fieldAttr *fa;
		if((ISigs.Sigs[InFields[i].firstsig].ignore & IGNORE_IRSIM) == 0) {
			fa = (struct fieldAttr *)ISigs.Fields[i].data;
			if(i>0)
				printf(",\n");
			if(fa && fa->vport_name) {
				printf("      .%s(i_%s)",
				       fa->vport_name, fa->vport_name);
			} else {
				for (j = 0; j < ISigs.Fields[i].nsigs; j++) {
					printf("      .");
					write_wire("", ISigs.Sigs[InFields[i].firstsig+j].name);
					putchar('(');
					write_wire("i_", ISigs.Sigs[InFields[i].firstsig+j].name);
					putchar(')');
					if(j<ISigs.Fields[i].nsigs-1)
						printf(",\n");
				}
			}
		}
	}
	for(i = 0; i < OSigs.nFields; i++) {
		struct fieldAttr *fa;
		fa = (struct fieldAttr *)OSigs.Fields[i].data;
		if(fa && fa->vport_name) {
			printf(",\n");
			printf("      .%s(o_%s)",
			       fa->vport_name, fa->vport_name);
		}
	}
	printf("\n   );\n\n");

	if(StatNumSigs) {
		printf("   initial begin\n");
		for (i = 0; i < StatNumSigs; i++) {
			printf("      force dut.%s = 1'b%d;\n",
			       StatSigs[i].name, (StatSigs[i].val & 1));
		}
		printf("   end\n");
	}
}


///////////////////////////////////////////////////////////////////////////
// Write code to set up for printing outputs
//////////////////////////////////////////////////////////////////////////

void
print_n(int n, char c)
{
	int i;
	for(i = 0; i < n; i++) {
		putchar(c);
	}
}

void
write_iosetup(int nopli, char *inname, char *outname, char *modulename)
{
	int nsp;
	int i, j;
	printf("   integer ofp;\n");
	printf("   integer ifp;\n");
	printf("   initial begin\n");
	printf("      $timeformat(-9, 5);\n");
	printf("      ofp = $fopen(\"%s\");\n", outname);
	printf("      if(ofp == 0) begin\n");
	printf("         $display(\"Failed to open %s - aborting\");\n", outname);
	printf("         $finish;\n");
	printf("      end // if (ofp == 0)\n");
	printf("      $fdisplay(ofp, \"Doverilog simulation of %s using %s\\n\");\n", modulename, inname);

	nsp = 0;
	for (i = 0; i < ISigs.nFields; i++) {
		for (j = 0; j < ISigs.Fields[i].nsigs; j++) {
			printf("      $fdisplay(ofp, \"");
			print_n(nsp, ' ');
			printf("%s\");\n", ISigs.Sigs[ISigs.Fields[i].firstsig+j].name);
			nsp++;
		}
		nsp++;
	}
	nsp++;

	for (i = 0; i < OSigs.nFields; i++) {
		for (j = 0; j < OSigs.Fields[i].nsigs; j++) {
			printf("      $fdisplay(ofp, \"");
			print_n(nsp, ' ');
			printf("%s\");\n", OSigs.Sigs[OSigs.Fields[i].firstsig+j].name);
			nsp++;
		}
		nsp++;
	}
	nsp++;

	for (i = 0; i < MonSigs.nFields; i++) {
		for (j = 0; j < MonSigs.Fields[i].nsigs; j++) {
			printf("      $fdisplay(ofp, \"");
			print_n(nsp, ' ');
			printf("%s\");\n", MonSigs.Sigs[MonSigs.Fields[i].firstsig+j].name);
			nsp++;
		}
		nsp++;
	}
	printf("      $fdisplay(ofp, \"");
	print_n(nsp, '=');
	printf("\");\n");

	if(nopli == 0) {
		printf("      ifp = $infopen(\"%s\", %d, %d, %d);\n", 
	       inname, ISigs.nSigs, OSigs.nSigs, InfileActiveEdge);
		printf("      if(ifp == 0) begin\n");
		printf("         $display(\"Failed to open %s - aborting\");\n", inname);
		printf("         $finish;\n");
		printf("      end // if (ifp == 0)\n");
	}
	printf("   end\n\n");
}

///////////////////////////////////////////////////////////////////////////
// Write code for clock generator
//////////////////////////////////////////////////////////////////////////
void
write_clockgen(int period)
{
	printf("   integer cycle;\n");
	printf("   initial begin\n");
	printf("      Clk = 1;\n");
	printf("      cycle = 0;\n");
	printf("   end\n");
	printf("   always begin\n");
	printf("      #%d Clk = ~Clk;\n", period);
	printf("       if(Clk == 1)\n");
	printf("          cycle = cycle + 1;\n");
	printf("   end\n\n");
}

///////////////////////////////////////////////////////////////////////////
// Write code to print outputs on each cycle
//////////////////////////////////////////////////////////////////////////
void
write_printblock(int period)
{
	int i,j;
	switch(InfileActiveEdge) {
	case EDGE_FALL:
		printf("   always @(posedge Clk) begin\n");
		break;
	case EDGE_RISE:
		printf("   always @(negedge Clk) begin\n");
		break;
	case EDGE_BOTH:
		printf("   always @(Clk) begin\n");
		break;
	}

	/* Waiting period-1 ticks is pretty crufty.
	 * What we really want is "$strobe-without-newline", but it doesn't
	 * exist.  Damn nonorthagonal standards
	 */
	printf("      #%d\n", period - 1);
	printf("      if(cycle >= 1) begin\n");
	
	for(i = 0; i < ISigs.nFields; i++) {
		for (j = 0; j < ISigs.Fields[i].nsigs; j++) {
			struct fieldAttr *fa;
			fa = (struct fieldAttr *)ISigs.Fields[i].data;
			if(fa && fa->vport_name) {
				printf("         $fwrite(ofp, \"%%b\", i_%s);\n",
				       ISigs.Sigs[ISigs.Fields[i].firstsig+j].name);
				       
			} else {
				if(ISigs.Sigs[ISigs.Fields[i].firstsig+j].ignore & IGNORE_IRSIM)
					printf("         $fwrite(ofp, \"%%b\", 1'bx);\n");
				else {
					printf("         $fwrite(ofp, \"%%b\", ");
					write_wire("i_", ISigs.Sigs[ISigs.Fields[i].firstsig+j].name);
					printf(");\n");
				}
			}
		}
		printf("         $fwrite(ofp, \" \");\n");
	}
	printf("         $fwrite(ofp, \" \");\n");
	for(i = 0; i < OSigs.nFields; i++) {
		struct fieldAttr *fa;
		fa = (struct fieldAttr *)OSigs.Fields[i].data;
		for (j = 0; j < OSigs.Fields[i].nsigs; j++) {
			if(fa && fa->vport_name) {
				printf("         $fwrite(ofp, \"%%b\", o_%s",
				       fa->vport_name);
				if(OSigs.Fields[i].nsigs > 1)
					printf("[%i]", OSigs.Fields[i].nsigs - 1 - j);
				printf(");\n");
				
			} else {
				printf("         $fwrite(ofp, \"%%b\", dut.%s);\n",
				       OSigs.Sigs[OSigs.Fields[i].firstsig+j].name);
			}
		}
		printf("         $fwrite(ofp, \" \");\n");
	}
	printf("         $fwrite(ofp, \" \");\n");
	for(i = 0; i < MonSigs.nFields; i++) {
		for (j = 0; j < MonSigs.Fields[i].nsigs; j++)
			printf("         $fwrite(ofp, \"%%b\", dut.%s);\n",
			       MonSigs.Sigs[MonSigs.Fields[i].firstsig+j].name);
		printf("         $fwrite(ofp, \" \");\n");
	}

	printf("         $fwrite(ofp, \" %%s %%0d\", ((Clk == 1) ? \"f\" : \"r\"), cycle);\n");
	printf("         $fwrite(ofp, \" #%%t\", $time);\n");
	printf("         $fwrite(ofp, \"\\n\");\n");
	printf("      end\n");
	printf("   end\n\n");
	
}

///////////////////////////////////////////////////////////////////////////
// Write appropriate @edge for stimulus reading/applying
//////////////////////////////////////////////////////////////////////////
void
write_edge()
{
	switch(InfileActiveEdge) {
	case EDGE_FALL:
		printf("@(negedge Clk)");
		break;
	case EDGE_RISE:
		printf("@(posedge Clk)");
		break;
	case EDGE_BOTH:
		printf("@(Clk)");
		break;
	}
}

///////////////////////////////////////////////////////////////////////////
// Write verilog testbench stimulus reader/applier
//////////////////////////////////////////////////////////////////////////
void
write_stimreader(int nopli)
{
	if(nopli) {
		char invec [MAX_SIGS];         // Variables for parsing .in file vectors
		char outvec [MAX_SIGS];
		int inflag, outflag;
		int in_edge;
		int vecno;
		int incycle;
		int ineof;
		int i;
		char *cp;
		char line [MAX_LINE_LEN];
		char *in_comment;

		printf("\n   initial begin\n");
		ineof = 0;
		vecno = 0;
		while (!ineof) {
			// Get next vector from .in file
			do {
				int status;
				status = parse_infile_vector(&inflag, invec, &outflag,
							  outvec, line, &in_edge, &in_comment);
				if (status ==  1) 
					ineof = 1;
				if (status == -1)
					infile_error(1, "syntax error in vectors section");
				// testbench will write comment lines to its output
				if (!inflag && !ineof) {
					if(cp = strchr(line, '\n'))
					   *cp = 0;
					printf("        $fdisplay(ofp, \"# %s\");\n", line);
				}

			} while (ineof == 0 && inflag == 0);
			vecno++;
			if(inflag) {
				printf("        ");
				if(vecno > 1) {
					write_edge();
				}
				if(ISigs.nSigs == 1) {
					printf(" inbits <= 2'b%c_x;\n",
					       invec[0]);
				} else {
					printf(" inbits <= %d'b", ISigs.nSigs);
					for(i = 0; i < ISigs.nSigs; i++) {
						putchar(invec[i]);
					}
					printf(";\n");
				}
				if(vecno == 1) {
					printf("        ");
					write_edge();  // initialization edge
					printf(";\n");
					if(InfileActiveEdge == EDGE_BOTH) {
						printf("        ");
						write_edge();
						printf(";\n");
					}
				}
			}
		}
		printf("        ");
		write_edge();
		printf(" $finish;\n");
	} else {
		printf("\n   integer r;\n");
		printf("   always ");
		write_edge();
		printf(" begin\n");
		printf("       r = $nextvec(ifp, inbits, outbits);\n");
		printf("       if(r < 0)\n");
		printf("          $finish;\n");
	}
	printf("   end\n");
}


///////////////////////////////////////////////////////////////////////////
// Write verilog to dump in vcd format
//////////////////////////////////////////////////////////////////////////
void write_dump(int type, char *fname, char *dut_name)
{
	switch(type) {
	case DUMP_VCD:
		printf("    initial begin\n");
		printf("        $dumpfile(\"%s.vcd\");\n", fname);
		printf("        $dumpvars(100,dut);\n");
		printf("    end\n\n");
	break;
	case DUMP_SHM:
		printf("    initial begin\n");
		printf("        $shm_open(\"%s.shm\");\n", fname);
		printf("        $shm_probe(\"AC\");\n");
		printf("    end\n\n");
	break;
	case DUMP_FSDB:
		printf("    initial begin\n");
		printf("        $fsdbDumpfile(\"%s.fsdb\");\n", fname);
		printf("        $fsdbDumpvars;\n");
		printf("    end\n\n");
	break;
	case DUMP_LXT:
		printf("	// blech: must set IVERILOG_DUMPER=lxt in environment to get LXT.\n");
		printf("    initial begin\n");
		printf("        $dumpfile(\"%s.lxt\");\n", fname);
		printf("        $dumpvars(100,dut);\n");
		printf("    end\n\n");
	break;
	}
}


///////////////////////////////////////////////////////////////////////////
// Write verilog testbench trailer
//////////////////////////////////////////////////////////////////////////
void
write_trailer(char *mod)
{
	printf("\nendmodule // %s\n", mod);
}


///////////////////////////////////////////////////////////////////////////
// callback for vport attribute
///////////////////////////////////////////////////////////////////////////
static void parse_attribute_vport(int id, char *value, Sig *sigt, Field *fld)
{
	struct fieldAttr *fa;

	if(fld->data == NULL) {
		fld->data = fa = new (struct fieldAttr);
		fa->vport_name = NULL;
	} else {
		fa = (struct fieldAttr *)fld->data;
	}

	if(fa->vport_name)
		infile_error(0, "duplicate vport attribute");
	else
		fa->vport_name = strdup(value);
}

///////////////////////////////////////////////////////////////////////////
// Main routine
//////////////////////////////////////////////////////////////////////////

int
main(int argc, char **argv)
{
	int c;
	char *cp;
	extern char *optarg;
	extern int optind;
	extern int optopt;
	// flags & arguments
	char *dut_name = NULL;
	char *testb_name = NULL;
	char *out_fname = NULL;
	char *dump_name = NULL;
	int got_clockname = 0;
	int nopli = 0;
	int period = 100;
	int dump_type = 0;
	char *infname;
	int err = 0;
	int powernets = 0;

	// parse arguments
	while((c = getopt(argc, argv, "a:c:d:gm:no:p:v")) != -1) {
		switch(c) {
		case 'a':
			if(strcasecmp(optarg, "vcd") == 0)
				dump_type = DUMP_VCD;
			else if(strcasecmp(optarg, "shm") == 0)
				dump_type = DUMP_SHM;
			else if(strcasecmp(optarg, "fsdb") == 0)
				dump_type = DUMP_FSDB;
			else if(strcasecmp(optarg, "lxt") == 0)
				dump_type = DUMP_LXT;
			else {
				fprintf(stderr, "%s: unknown waveform dump type %s\n", progname, optarg);
				err = 1;
			}
			break;

		case 'c':
			if(got_clockname == 0) {
				clockName = optarg;
				got_clockname = 1;
			} else {
				extraClkNames[n_extraClks++] = optarg;
			}
			break;
		case 'd':
			dut_name = optarg;
			break;
		case 'g':
			powernets = 1;
			break;
		case 'm':
			testb_name = optarg;
			break;
		case 'n':
			nopli = 1;
			break;
		case 'o':
			out_fname = optarg;
			break;
		case 'p':
			period = atoi(optarg);
			break;
		case 'v':
			g_verbose = 1;
			break;
		default:
			fprintf(stderr, "%s: Unrecognized option: -%c\n", progname, optopt);
			err = 2;
			break;
		}
	}
	if(optind + 1 != argc) {
		fprintf(stderr, "%s: input file not specified.\n", progname);
		err = 3;
	}
	if(err) {
		usage();
		exit(err);
	}
	infname = argv[optind];
	InfileActiveEdge = 0;

	// Parse infile header
	set_attribute_parser(SATTR_VPORT, parse_attribute_vport);
	parse_infile_header(infname);

	// generate module and file names if not specified
	if(!dut_name) {
		dut_name = strdup(infname);
		cp = strchr(dut_name, '.');
		if(cp)
			*cp = 0;
	}
	if(!testb_name) {
		testb_name = new char[strlen(dut_name)+10];
		strcpy(testb_name, dut_name);
		strcat(testb_name, "_runtest");
	}
	if(!out_fname) {
		out_fname = new char[strlen(dut_name)+10];
		strcpy(out_fname, dut_name);
		strcat(out_fname, ".vout");
	}
	dump_name = new char [strlen(infname)+10];
	strcpy(dump_name, infname);
	cp = strrchr(dump_name, '.');
	if(cp)
		*cp = 0;

	write_header(testb_name);
	write_decls(powernets);
	write_minst(dut_name, powernets);
	write_iosetup(nopli, infname, out_fname, dut_name);
	write_clockgen(period);
	write_printblock(period);
	if(dump_type)
		write_dump(dump_type, dump_name, dut_name);
	write_stimreader(nopli);
	write_trailer(testb_name);

	exit(0);
}
