/* 
 *
 * infile.h -- parsing routines for doirsim/dospice/doverilog ".in" file
 *
 * by Steve Tell
 */
#include <glib.h>

#define MAX_ITERATORS       4	/* number of iterators in a signal name    */

// Mnemonic defines for default fanout and delay values for signals
#define DEFAULT_FANOUT     10
#define DEFAULT_DELAY       0

/* Mnemonic defines for ignore flags (signals to ignore in comparisons)
// IGNORE on an input also means to not drive the input into the DUT at
// all, and record the input as "x" in the output file. */
#define IGNORE_NONE       0x0	/* default                                 */
#define IGNORE_DELAY      0x1	/* ignore setup times in analog simulation   */
#define IGNORE_SPICE      0x2	/* ignore values in analog simulation        */
#define IGNORE_CAZM       IGNORE_SPICE
#define IGNORE_IRSIM      0x4	/* ignore values in irsim sim.	           */
#define IGNORE_ALL        0x7	/* ignore values in all simulations.    */

/* defines for signal edge style */
#define EDGE_DEFAULT	0
#define EDGE_FALL	0x1	/* default. falling edge of clock */
#define EDGE_RISE	0x2	/* rising edge */
#define EDGE_BOTH	(EDGE_FALL|EDGE_RISE)	/* both edges */

/* structures used while parsing a .in file */

enum sigtype {UNKNOWN=0, STATSIG=1, INSIG=2, OUTSIG=4, MONSIG=8, CLOCKSIG=16};

struct inf_signal {
	enum sigtype type;
	int index;		/* index of value in vector arrays */
	char    *name;
	int     val;		/* valid for type=STATSIG only */
	float   fanout;
	float   delay;
	int     ignore;
	int     edgestyle;
	char    *wavename;
};

struct inf_field {
	char    *name;
	int     firstsig;
	int     nsigs;
	char *probestr; /* string used in probe statements to print whole field*/
	/* formatting information.  
	// non-binary I/O is not fully supported yet. */
	int width;	/* width of printed output field */
	char conv;	/* 'b', 'h', 'o', 'd' */
	int leadzero;	/* true=leading zeros, else spaces */
	void *data;	/* pointer for user structure. */
};

struct inf_sigset {
	GPtrArray *sigs;	/* elements are struct inf_signal*  */
	GPtrArray *fields;  /* elelements are struct inf_field*    */
	GHashTable *sighash;
};

struct infile {
	FILE *fp;
	char *filename;
	int lineno;
	char *linebuf;
	int lbufsize;
	struct inf_sigset statics;
	struct inf_sigset inputs;
	struct inf_sigset outputs;
	struct inf_sigset monitors;
	int ActiveEdge; /* which edges get used; OR of edgestyleof all InSigs and OutSigs */
	void *userdata;
};

#ifndef __GNUC__
#define __attribute__(a)
#endif

/* Procedure declarations */
void infile_error(struct infile *inf, int die, char *fmt, ...) 
	__attribute__((format (printf, 3, 4)));

extern struct infile *infile_new(char *filename);
extern int infile_parse_header(struct infile *inf);
extern int infile_vector(struct infile *inf, int *inflag,  char *invec,
			 int *outflag, char *outvec, int *edgep, 
			 char **commentp);
void infile_close(struct infile *inf);

extern int infile_num_vectors(struct infile *inf, int outspresent);

/* callback for InFile signal-field attributes */
typedef void (*ATTRCB)(struct infile *inf, int id, char *value, 
		       struct inf_signal *sig, struct inf_field *fld);
extern void infile_set_attribute_parser(int id, ATTRCB afunc);

// attribute IDs 
// All possible attributes are listed here (and in inparse.C)
// so we can ignore known ones even if we don't parse them for a particular
// client of inparse.
#define SATTR_ETFO	1
#define SATTR_IGNORE	2
#define SATTR_FORMAT	3
#define SATTR_VPORT	4
#define SATTR_PROBE	5

/* callback mechanism for InFile header sections */
typedef void (*INSCB)(struct infile *inf, int id, char *line);
extern void infile_set_section_handler(int id, INSCB afunc);

#define INSECT_UNKNOWN	0
#define INSECT_STATIC	1
#define INSECT_INPUTS	2
#define INSECT_OUTPUTS	3
#define INSECT_MONITOR	4
#define INSECT_VECTORS	5
/* new ones not fully supported yet */
#define INSECT_CLOCKS	6
#define INSECT_SPICE	7
#define INSECT_IRSIM	8
#define INSECT_VERILOG	9
