/*
//
// inhdump.C -- debugging utility to dump out info from a .in file
//    as read by inparse routines.
//
// $Log: inhdump.c,v $
// Revision 1.2  2002/11/14 21:10:08  cvs
// remove some debug stuff from inhdump
//
// Revision 1.1  2002/11/08 20:42:20  cvs
// write transcribe_vectors to finish incat
// fetch and submit new inhdump
// produce error messgae instead of coredump for unknown infile sections
//
// Revision 1.2  1999/11/01 22:26:40  tell
// Dump out probe attribute if present.  minor changes to accomodate table-based
// attribute scheme.
//
// Revision 1.1  1999/11/01 20:02:00  tell
// Initial revision
//
// Revision 1.2  1998/06/02 19:06:13  tell
// Added attribute-callback to dump additional attributes
// changed input and output signal access to use new SigSet structure instead of
// seperate globals
//
// Revision 1.1  1998/05/29 17:38:55  tell
// Initial revision
//
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "infile.h"

static char *progname = "in2ir";
static int g_verbose;

static void 
usage()
{
	fprintf(stderr, "Usage:  %s [options] <infile>\n", progname);
	fprintf(stderr, "        <infile>     is a doirsim .in file\n");
	fprintf(stderr, "  options:\n");
	fprintf(stderr, "        -v           verbose mode\n");

	exit(3);
}

int
main(int argc, char **argv)
{
	int i, j;
	int c;
	extern char *optarg;
	extern int optind;
	extern int optopt;
	struct infile *inf;
	struct inf_signal *sig;
	struct inf_field *field;
	gpointer gp;

	// parse arguments
	while((c = getopt(argc, argv, "v")) != -1) {
		switch(c) {
		case 'v':
			g_verbose = 1;
			break;
		default:
			fprintf(stderr, "%s: Unrecognized option: -%c\n", progname, optopt);
			usage();
			break;
		}
	}
	if(optind + 1 != argc) {
		fprintf(stderr, "%s: input file not specified.\n", progname);
		usage();
	}

	/* open .in file and parse the header */
	inf = infile_new(argv[optind]);
	if(!inf) {
		perror(argv[optind]);
		exit(1);
	}
	if(!infile_parse_header(inf)) {
		fprintf(stderr, "%s: bad header\n");
		exit(1);
	}

	/* Print out static signals */
	for (i = 0; i < inf->statics.sigs->len; i++) {
		sig =  (struct inf_signal *)
			g_ptr_array_index(inf->statics.sigs, i);
		printf("static %s %d\n", sig->name, sig->val);
	}

	/* inputs */
	for (i = 0; i < inf->inputs.fields->len; i++) {
		field =  (struct inf_field *)
			g_ptr_array_index(inf->inputs.fields, i);

		printf("input field %s (%d bits, width=%d)\n", 
		       field->name, field->nsigs, field->width);
		for (j = 0; j < field->nsigs; j++) {
			sig =  (struct inf_signal *)
				g_ptr_array_index(inf->inputs.sigs, field->firstsig+j);
			printf("  %s [%d,%d]\n", sig->name, field->firstsig+j,
			       sig->index);
		}
	}
	if(gp = g_hash_table_lookup(inf->inputs.sighash, "b06")) {
		sig =  (struct inf_signal *)gp;
		printf(" b06 by hash got %s [%d] type=%d\n", 
		       sig->name, sig->index, sig->type);
	} else {
		printf(" b06 not found in hash\n");
	}

	/* outputs */
	for (i = 0; i < inf->outputs.fields->len; i++) {
		field =  (struct inf_field *)
			g_ptr_array_index(inf->outputs.fields, i);

		printf("output field %s (%d bits)", field->name, field->nsigs);
		if(field->probestr)
			printf(" probe=\"%s\"", field->probestr);
		putchar('\n');
		for (j = 0; j < field->nsigs; j++) {
			sig =  (struct inf_signal *)
				g_ptr_array_index(inf->outputs.sigs, field->firstsig+j);
			printf("  %s [%d]\n", sig->name, sig->index);
		}
	}

	// monitor
	for (i = 0; i < inf->monitors.fields->len; i++) {
		field =  (struct inf_field *)
			g_ptr_array_index(inf->monitors.fields, i);

		printf("monitor field %s (%d bits)", field->name, field->nsigs);
		if(field->probestr)
			printf(" probe=\"%s\"", field->probestr);
		putchar('\n');
		for (j = 0; j < field->nsigs; j++) {
			sig =  (struct inf_signal *)
				g_ptr_array_index(inf->monitors.sigs, field->firstsig+j);
			printf("  %s [%d]\n", sig->name, sig->index);
		}
	}
	

	exit(0);
}

