///////////////////////////////////////////////////////////////////////////
//
// spice2out -- Compute .out file based on waveforms from a 
//		spice analog simulation run.
//             Outputs are tested against ETFO-style setup times 
//             from .in file.
//////////////////////////////////////////////////////////////////////////

/*
 * Copyright 1994-1999  Stephen G. Tell
 * Copyright 1994,1995  Steven Molnar, John Eyles
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <stdarg.h>
#include <ctype.h>
#include <float.h>
#include <limits.h>
#include "inparse.h"
#include "vec.h"
#include "spicestream.h"

static char *progname = "spice2out";

//
// Mnemonic defines
//

#define TIMING_EPSILON	DBL_EPSILON

#define DEFAULT_ETFO_WDIR "/usr/msl/vlsi/lib/pxfl"
#define DEFAULT_ETFO_CASE "W"
#define DEFAULT_ETFO_TYPE "etff"

char edge_chars[] = "?FRB";

// Voltage thresholds (expressed as fraction of vsupply)
#define VTHRESHOLD_TOL   0.2

// Sampling voltage for ETFO signals (as fraction of vsupply)
#define VSAMPLE_FRACT   0.2

// Sample types
#define FALL_L      0
#define FALL_H      1
#define RISE_L      2
#define RISE_H      3

#define MAX_ETFO_TABLES	10	// max number of different etff wave tables

//
// Global variable defines
//

int g_verbose_err = 0;
int g_debug = 0;

// clock period can be specified on the command line.
int do_clock_period_override = 0;
double clock_period_override;

// Active edge of clock
int ActiveEdge;

// Sampling voltage for active edge of clock
double VSampX;

// Info from one etff file.
struct EtffData {
	char *name;
	// Interpolated times from etff model for signal 'h' and 'l' events 
	// (each indexed by sample type defined above).  ta is unloaded,
	// tb is with fanout of 10.
	double ta[4];
	double tb[4];
	double clock_period;	// in ns
	double Vih, Vil;	// input high and low thresholds
	double Voh, Vol;	// output high and low levels
	double Vsamp;		// single threshold when used as clock
	int Vsamp_valid;
};

class EtffList {
	char *etff_wdir;
	char *etff_case;
	struct EtffData **tablist;
	int ntables;
	
	EtffData *read_etff_file(char *name, char *dir, char *fcase);
public:
	EtffList(char *dir, char *ccase);

	EtffData *find(char *name);
};

EtffList *etff_list;

// info needed to check timing on each output signal.
typedef struct SigCheckStruct {
	double time_last_X;	// time of last transition to valid value
	double setup_H;		// time signal must be 1 before clock edge
	double setup_L;		// time signal must be 0 before clock edge
	int do_hold;
	double hold_H;		// time signal must stay 1 after clock edge
	double hold_L;		// time signal must stay 0 after clock edge
	EtffData *edata;
	int holderr;
} SIGCHK;

// one for each signal, same indices as OutSigs.
SIGCHK *SigChk;

// info for each input or monitor signal
typedef struct INSIGChk {
	EtffData *edata;
};
INSIGChk  *InSigChk;
INSIGChk  *MonSigChk;

// info on clock waveforms
typedef struct ClockInfo {
	char *name;
	double Vmin;
	double Vmax;
	double Vsamp;
	EtffData *edata;
};

// Waveform file information
static SpiceStream *WFile;

// Value of signals at current and previous step
double ClkValue;
double LastClkValue;
double InSigValue     [MAX_SIGS];
double LastInSigValue [MAX_SIGS];
double OutSigValue    [MAX_SIGS];
double LastOutSigValue[MAX_SIGS];
double MonSigValue    [MAX_SIGS];
double LastMonSigValue[MAX_SIGS];

///////////////////////////////////////////////////////////////////////////
// Convert voltage to logic level.  used often, so factored out here.
///////////////////////////////////////////////////////////////////////////
char logic_thresh(struct EtffData *ed, double voltage)
{
	if(voltage > ed->Vih)
		return '1';
	else if(voltage < ed->Vil)
		return '0';
	else
		return 'x';
}

///////////////////////////////////////////////////////////////////////////
// Initialize list of etff wave tables
///////////////////////////////////////////////////////////////////////////
EtffList::EtffList(char *dir, char *fcase)
{
	if(dir)
		etff_wdir = strdup(dir);
	else
		etff_wdir = DEFAULT_ETFO_WDIR;
	if(fcase)
		etff_case = strdup(fcase);
	else
		etff_case = DEFAULT_ETFO_CASE;

	tablist = new EtffData*[MAX_ETFO_TABLES];
	ntables = 0;
}

EtffData *EtffList::
find(char *name)
{
	int i;
	for(i = 0; i < ntables; i++) {
		if(strcmp(name, tablist[i]->name)==0)
			return tablist[i];
	}
	// not found: try to load it
	struct EtffData *newtab;
	newtab = read_etff_file(name, etff_wdir, etff_case);
	if(newtab) {
		tablist[ntables++] = newtab;
	}
	return newtab;
}


///////////////////////////////////////////////////////////////////////////
// Parse etff model file, loading etff vars
//////////////////////////////////////////////////////////////////////////
EtffData *EtffList::
read_etff_file(char *name, char *dir, char *fcase)
{
	// Maximum number of datapoints in lo-to-hi or hi-to-lo transition
#define MAX_ETFF_SAMPLES 128

	// etff model variables
	double tz;                     // Time clock = vdd/2 during hi-to-lo transition 
	int nd;			// Number of falling edge samples
	double vd [MAX_ETFF_SAMPLES];	// Falling edge sample voltages/times
	double tda[MAX_ETFF_SAMPLES];
	double tdb[MAX_ETFF_SAMPLES];
	int nu;			// Number of rising edge samples
	double vu [MAX_ETFF_SAMPLES];	// Rising edge sample voltages/times
	double tua[MAX_ETFF_SAMPLES];
	double tub[MAX_ETFF_SAMPLES];

	// other vars
	char line[MAX_LINE_LEN];
	char fname[1024];
	FILE       *mfile;
	int lineno = 0;
	int got_Vih = 0, got_Vil = 0, got_Voh = 0, got_Vol = 0;
	int got_tz = 0;
	int err = 0;
	int i;

	sprintf(fname, "%s/%s.%s", dir, name, fcase);
	if ((mfile = fopen(fname, "r")) == NULL) {
		perror(fname);
		return NULL;
	}
	fprintf(stderr, "%s: Loading ETFO file %s\n", progname, fname);

	EtffData *ntab = new EtffData;
	ntab->name = strdup(name);

	char *keywd;
	char *value;
	double fval;
	// Skip down to 'down' section, parsing name=value parameters
	while(1) {
		// Read a line
		if (fgets(line, MAX_LINE_LEN, mfile) == NULL) {
			fprintf(stderr, "Premature end of file in ETFO file %s\n", fname);
			exit(1);
		}
		lineno++;
		keywd = strtok(line, " \t\n=");
		if(!keywd) {
			fprintf(stderr, "%s:%d: Syntax Error; keyword expected.\n", fname, lineno);
			exit(1);
		}
		if(strcmp(keywd, "down")==0)
			break;
		value = strtok(NULL, " \t\n=");
		if(!value) {
			fprintf(stderr, "%s:%d: Syntax Error; parameter value expected.\n", fname, lineno);
			exit(1);
		}
		if(sscanf(value, "%lf", &fval) != 1) {
			fprintf(stderr, "%s:%d: Syntax Error; floating-point parameter value expected.\n", fname, lineno);
			exit(1);
		}

		// parsed it all; now check for each attribute keyword
		if(strcmp(keywd, "tz")==0) {
			tz = fval * 1.0e9;
			got_tz = 1;
		} else if(strcmp(keywd, "pw")==0) {
			ntab->clock_period = fval * 1.0e9;
		} else if(strcasecmp(keywd, "Vih")==0) {
			ntab->Vih = fval;
			got_Vih = 1;
		} else if(strcasecmp(keywd, "Voh")==0) {
			ntab->Voh = fval;
			got_Voh = 1;
		} else if(strcasecmp(keywd, "Vil")==0) {
			ntab->Vil = fval;
			got_Vil = 1;
		} else if(strcasecmp(keywd, "Vol")==0) {
			ntab->Vol = fval;
			got_Vol = 1;
		} else if(strcasecmp(keywd, "Vsamp")==0) {
			ntab->Vsamp = fval;
			ntab->Vsamp_valid = 1;
		} 
	}

	// Parse down entries
	nd = 0;
	while(1) {
		if (fgets(line, MAX_LINE_LEN, mfile) == NULL) {
			fprintf(stderr, "Premature end of file in ETFO file %s\n", fname);
			exit(1);
		}
      
		if (strstr(line, "up") != 0)
			break;

		if (sscanf(line, "%lf %lf %lf\n", &vd[nd], &tda[nd], &tdb[nd]) != 3) {
			fprintf(stderr, "Error in down section of ETFO file %s\n", fname);
			exit(1);
		}
		nd++;
		if(nd >= MAX_ETFF_SAMPLES) {
			fprintf(stderr, "%s: too many samples in down section; increase MAX_ETFF_SAMPLES\n", fname);
		}
	}
	if(nd < 2) {
		fprintf(stderr, "%s: not enough entries in down section (2 required)\n", fname);
		exit(1);
	}

	// Parse up entries
	nu = 0;
  	while (1) {
		if (fgets(line, MAX_LINE_LEN, mfile) == NULL)
			break;

		if (sscanf(line, "%lf %lf %lf\n", &vu[nu], &tua[nu], &tub[nu]) != 3) {
			fprintf(stderr, "Error in up section of ETFO file %s\n", fname);
			exit(1);
		}
		nu++;
		if(nu >= MAX_ETFF_SAMPLES) {
			fprintf(stderr, "%s: too many samples in up section; increase MAX_ETFF_SAMPLES\n", fname);
		}
	}
	if(nu < 2) {
		fprintf(stderr, "%s: not enough entries in up section (2 required)\n", fname);
		exit(1);
	}
	fclose(mfile);

	if(!got_Voh)
		ntab->Voh = vd[0];
	if(!got_Vol)
		ntab->Vol = vu[0];
	if(!got_Vih) {
		double Vswing = ntab->Voh - ntab->Vol;
		ntab->Vih = ntab->Vol + Vswing*(1.0-VTHRESHOLD_TOL);
	}
	if(!got_Vil) {
		double Vswing = ntab->Voh - ntab->Vol;
		ntab->Vil = ntab->Vol + Vswing*VTHRESHOLD_TOL;
	}

	// check Vxx for consistency
	// lots of things assume Vol <= Vil < Vih <= Voh
	if(!(ntab->Vol <= ntab->Vil)) {
		fprintf(stderr, "%s: Voltage inconsistency: Vol=%f Vil=%f\n",
			fname, ntab->Vol, ntab->Vil);
		err=1;
	}
	if(!(ntab->Vih <= ntab->Voh)) {
		fprintf(stderr, "%s: Voltage inconsistency: Vih=%f Voh=%f\n",
			fname, ntab->Vih, ntab->Voh);
		err=1;
	}
	if(!(ntab->Vil < ntab->Vih)) {
		fprintf(stderr, "%s: Voltage inconsistency: Vil=%f Vih=%f\n",
			fname, ntab->Vil, ntab->Vih);
		err=1;
	}
	if(err)
		exit(1);

  // Interpolate times and reference them to tz
  for (i = 1; i < nd; i++)
    if (vd[i-1] >= ntab->Vil && ntab->Vil >= vd[i]) {
	ntab->ta[FALL_L] = -tz + tda[i-1] + 
		(tda[i] - tda[i-1]) * (ntab->Vil-vd[i-1]) / (vd[i]-vd[i-1]);

	ntab->tb[FALL_L] = -tz + tdb[i-1] + 
		(tdb[i] - tdb[i-1]) * (ntab->Vil-vd[i-1]) / (vd[i]-vd[i-1]);
    }

  for (i = 1; i < nd; i++)
    if (vd[i-1] >= ntab->Vih && ntab->Vih >= vd[i]) {
	ntab->ta[FALL_H] = -tz + tda[i-1] + 
		(tda[i] - tda[i-1]) * (ntab->Vih-vd[i-1]) / (vd[i]-vd[i-1]);
	ntab->tb[FALL_H] = -tz + tdb[i-1] +
		(tdb[i] - tdb[i-1]) * (ntab->Vih-vd[i-1]) / (vd[i]-vd[i-1]);
    }

  for (i = 1; i < nu; i++)
    if (vu[i-1] <= ntab->Vil && ntab->Vil <= vu[i]) {
	ntab->ta[RISE_L] = -tz + tua[i-1] +
		(tua[i] - tua[i-1]) * (ntab->Vil-vu[i-1]) / (vu[i]-vu[i-1]);
	ntab->tb[RISE_L] = -tz + tub[i-1] +
		(tub[i] - tub[i-1]) * (ntab->Vil-vu[i-1]) / (vu[i]-vu[i-1]);
    }

  for (i = 1; i < nu; i++)
    if (vu[i-1] <= ntab->Vih && ntab->Vih <= vu[i]) {
	ntab->ta[RISE_H] = -tz + tua[i-1] +
		(tua[i] - tua[i-1]) * (ntab->Vih-vu[i-1]) / (vu[i]-vu[i-1]);
	ntab->tb[RISE_H] = -tz + tub[i-1] +
		(tub[i] - tub[i-1]) * (ntab->Vih-vu[i-1]) / (vu[i]-vu[i-1]);
    }

	if(g_debug) {
		fprintf(stderr, "ETFO wave %s: Vih=%.3f vil=%.3f\n", 
			name, ntab->Vih, ntab->Vil);
		fprintf(stderr, "%s->ta[FALL_L] = %f\n", name, ntab->ta[FALL_L]);
		fprintf(stderr, "%s->ta[FALL_H] = %f\n", name, ntab->ta[FALL_H]);

		fprintf(stderr, "%s->tb[FALL_L] = %f\n", name, ntab->tb[FALL_L]);
		fprintf(stderr, "%s->tb[FALL_H] = %f\n", name, ntab->tb[FALL_H]);
  
		fprintf(stderr, "%s->ta[RISE_L] = %f\n", name, ntab->ta[RISE_L]);
		fprintf(stderr, "%s->ta[RISE_H] = %f\n", name, ntab->ta[RISE_H]);

		fprintf(stderr, "%s->tb[RISE_L] = %f\n", name, ntab->tb[RISE_L]);
		fprintf(stderr, "%s->tb[RISE_H] = %f\n", name, ntab->tb[RISE_H]);
	}

	return ntab;
}

//////////////////////////////////////////////////////////////////////////
// Read clock file.  (new and reasonably sane format.)
// A clock file is just an Etff waveform specification file.   
// The rising and falling segments were used to put together the clock
// waveform that the simulation was run with.
//
// old/new file compatibilty note:
// 	Vol and Voh in a wavefile are defined to be the nominal high
//	and low values, usually GND and Vdd.  
//	In the old files, we took Vsamp to be the midpoint of the
//	minimum and maximum excursion of the clock, including any overshoot.
//	Using the midpoint of the nominal high/low values is probably
//	more appropriate, but there may be a very slight change in results.
//	If any of this is a problem, you can always set Vsamp 
//	explicitly in the Clock file
//////////////////////////////////////////////////////////////////////////
ClockInfo *
read_clock_file(char *name)
{
	EtffData *ed;
	ed = etff_list->find(name);
	if(!ed) {
		return NULL;
	}

	ClockInfo *ci = new ClockInfo;
	ci->name = strdup(name);
	ci->edata = ed;

	ci->Vmin = ed->Vol;
	ci->Vmax = ed->Voh;
	if(ed->Vsamp_valid) {
		ci->Vsamp = ed->Vsamp;
	} else {
		ci->Vsamp = ( ci->Vmin + ci->Vmax ) / 2;
	}
	if(g_debug) {
		fprintf(stderr, "Clock waveform %s: min=%f max=%f samp=%f (new format)\n",
			ci->name, ci->Vmin, ci->Vmax, ci->Vsamp);
	}

	return ci;
}

//////////////////////////////////////////////////////////////////////////
// read old-format clock file and pick out necessary info.
// The old format is a CAZM-format "wave" statement with optional
// spice comments that can contain additional info.
// We look for min and max voltage levels, and
// special comments.  We don't even save the data points.
// The whole goal is to pick a sampling voltage at the halfway point.
//////////////////////////////////////////////////////////////////////////
ClockInfo *
read_old_clock_file(char *name, char *dir, char *fcase)
{
	char line[MAX_LINE_LEN];
	char fname[1024];
	FILE       *mfile;
	int lineno = 0;
	int got_vsamp = 0;

	sprintf(fname, "%s/%s.%s", dir, name, fcase);
	if ((mfile = fopen(fname, "r")) == NULL) {
		perror(fname);
		return NULL;
	}
	fprintf(stderr, "%s: Loading Clock file %s\n", progname, fname);

	ClockInfo *ci = new ClockInfo;
	ci->name = strdup(name);
	ci->edata = NULL;

	char *keywd;
	char *value;
	double fval;
	// Look for wave statement.  Parse special-comments 
	while(1) {
		// Read a line
		if (fgets(line, MAX_LINE_LEN, mfile) == NULL) {
			fprintf(stderr, "Premature end of file in ETFO file %s\n", fname);
			exit(1);
		}
		lineno++;
		if(strstr(line, "wave"))
			break;

		if(line[0] == '*' && line[1] == '*') {
			keywd = strtok(&line[2], " \t\n=");
			value = strtok(NULL, " \t\n=");
			if(keywd && value) {
				sscanf(value, "%lf", &fval);
			}
			if(strcasecmp(keywd, "Vsamp")==0) {
				ci->Vsamp = fval;
				got_vsamp = 1;
			}
			
		}
	}
	
	ci->Vmin = 1000;
	ci->Vmax = -1000;

	// we now have the wave line; process it.
	value = strchr(line, '{');	// first time
	if(!value) {
		fprintf(stderr, "%s:%d: no { on wave piecewise line\n", fname, lineno);
		exit(2);
	}
	value = strtok(value+1, " \t\n,}");	// first time
	double vclk;
	while(value) {
		value = strtok(NULL, " \t\n,}");	// voltage
		if(!value)
			break;
		sscanf(value, "%lf", &vclk);
		if(vclk > ci->Vmax)
			ci->Vmax = vclk;
		if(vclk < ci->Vmin)
			ci->Vmin = vclk;
		value = strtok(NULL, " \t\n,}");	// next time
	}
	if(!got_vsamp) {
		ci->Vsamp = (ci->Vmin + ci->Vmax)/2;
	}
	if(g_debug) {
		fprintf(stderr, "Clock waveform %s: min=%f max=%f samp=%f\n",
			ci->name, ci->Vmin, ci->Vmax, ci->Vsamp);
	}
	return ci;
}

///////////////////////////////////////////////////////////////////////////
// Construct info needed to check level and timing of each signal.
//////////////////////////////////////////////////////////////////////////
static void build_checkinfo()
{
	int i;
	int err = 0;

	// Build the sample array
	SigChk = new SIGCHK[OutNumSigs];  
	struct EtffData *ed;

	for (i = 0; i < OutNumSigs; i++) {
		char *wavename;
		if(OutSigs[i].wavename)
			wavename = OutSigs[i].wavename;
		else
			wavename = DEFAULT_ETFO_TYPE;
		ed = etff_list->find(wavename);
		if(!ed) {
			fprintf(stderr, "can't find ETFO waveform table for %s.\n",
				wavename);
			exit(2);
		}

		// general initialization
		SigChk[i].time_last_X = 0.0;
		SigChk[i].holderr = 1;
		SigChk[i].do_hold = 0;
		SigChk[i].edata = ed;

		// these are correct but not useful;
		// we need to get a seperate delay number to compute a
		// reasonable hold-time spec

		SigChk[i].hold_L = OutSigs[i].delay + ed->ta[RISE_L] 
			+ OutSigs[i].fanout*((ed->tb[RISE_L]-ed->ta[RISE_L])/10);

		SigChk[i].hold_H =  OutSigs[i].delay + ed->ta[FALL_H]
			+ OutSigs[i].fanout*((ed->tb[FALL_H]-ed->ta[FALL_H])/10);

		
		// ta and ed->tb are in terms of max delay from clock edge;
		// subtract from clock edge spacing to get min setup time
		// before edge.
		// Edge spacing is clock period for single-edge signals, and
		// half of period for double-edge.
		double edge_spacing;
		double period;
		if(do_clock_period_override) {
			period = clock_period_override;
		} else {
			period = ed->clock_period;
		}
		if(OutSigs[i].edgestyle == EDGE_BOTH)
			edge_spacing = period/2;
		else
			edge_spacing = period;


		SigChk[i].setup_L = edge_spacing -
		    (OutSigs[i].delay + ed->ta[FALL_L] +
		     OutSigs[i].fanout*((ed->tb[FALL_L]-ed->ta[FALL_L])/10) );

		SigChk[i].setup_H = edge_spacing -
		    (OutSigs[i].delay + ed->ta[RISE_H] +
		     OutSigs[i].fanout*((ed->tb[RISE_H]-ed->ta[RISE_H])/10));

		if(g_debug) {
			fprintf(stderr, "SigChk[%s].setup_H=%f\n", 
				OutSigs[i].name, SigChk[i].setup_H);
			fprintf(stderr, "SigChk[%s].setup_L=%f\n",
				OutSigs[i].name, SigChk[i].setup_L);
			fprintf(stderr, "SigChk[%s].hold_H=%f\n",
				OutSigs[i].name, SigChk[i].hold_H);
			fprintf(stderr, "SigChk[%s].hold_L=%f\n",
				OutSigs[i].name, SigChk[i].hold_L);
		}

		// Make sure it makes sense: setup + hold < edge spacing

		// signal falling
		if(SigChk[i].hold_H + SigChk[i].setup_L > edge_spacing) {
			fprintf(stderr, "Error: impossible constraints for signal %s; hold_H=%f, setup_L=%f\n",
				OutSigs[i].name,
				SigChk[i].hold_H, SigChk[i].setup_L);
			err=1;
			if(g_debug)
				abort();
		}
		// signal rising
		if(SigChk[i].hold_L + SigChk[i].setup_H > edge_spacing) {
			fprintf(stderr, "Error: impossible constraints for signal %s; hold_L=%f, setup_H=%f\n",
				OutSigs[i].name,
				SigChk[i].hold_L, SigChk[i].setup_H);
			err=1;
			if(g_debug)
				abort();
		}
	}

	if(err) {
		exit(2);
	}


	// Build the check-data arrays for input and monitor signals
	InSigChk = new INSIGChk[InNumSigs];  

	for (i = 0; i < InNumSigs; i++) {
		char *wavename;
		if(InSigs[i].wavename)
			wavename = InSigs[i].wavename;
		else
			wavename = DEFAULT_ETFO_TYPE;
		ed = etff_list->find(wavename);
		if(!ed) {
			fprintf(stderr, "can't find ETFO waveform table for %s.\n",
				wavename);
			exit(2);
		}

		InSigChk[i].edata = ed;
	}

	if(MonSigs.nSigs)
		MonSigChk = new INSIGChk[MonSigs.nSigs];  

	for (i = 0; i < MonSigs.nSigs; i++) {
		char *wavename;
		if(MonSigs.Sigs[i].wavename)
			wavename = MonSigs.Sigs[i].wavename;
		else
			wavename = DEFAULT_ETFO_TYPE;
		ed = etff_list->find(wavename);
		if(!ed) {
			fprintf(stderr, "can't find ETFO waveform table for %s.\n",
				wavename);
			exit(2);
		}

		MonSigChk[i].edata = ed;
	}
}


///////////////////////////////////////////////////////////////////////////
// Parse one cycle's worth of waveform information from WaveFile.
// Report vector of signal values in wavevec and delay flags in delayvec.
//
// On exit:
//   invec    = array consisting of [01X] for each input signal
//              (actually sampled at clock edge as sanity check)
//   outvec   = array consisting of [01X] for each output signal
//              (sampled at clock edge)
//   delayvec = array of delay flags, one for each output signal
//              ' ': signal met ETFO setup time (given final value)
//              'd': signal did not meet ETFO setup time
//   monvec   = array consisting of [01X] for each monitored signal
//              (sampled at clock edge)
//   time     = time (in ns) of active edge at which signals were sampled
//   edgetype = type of active edge that occured
//   report   = 1 if really want to check, 0 if we don't care.
//   slacktime = minimum slack time (in ns) over all output signals.
//             slack is positive if timing constraint met, negative if not.
//////////////////////////////////////////////////////////////////////////
int parse_wave_vector(char *invec, char *outvec, char *delayvec, char *monvec,
		      double *time, int inputs, int *edgetype, int report,
		      double *slacktime)
{
	static double Time;                 // Time at current waveform sample
	static double LastTime;	      // Time at previous waveform sample
	static double EdgeTime = 0.0;	      // Time at last active edge
	static double LastClkRise = 0.0;	// time of last rising clock 
	static double LastClkFall = 0.0;	// time of last falling clock
	static int    firstcall    = 1;     // Flag set during first call of routine
	int    cursamp;
	int    fnd_edge;
	int    i;
	double DeltaT;
	double Values[MAX_SIGS];
	int min_slack_sig;
	double min_slack;
	double sig_slack;

	// Process lines until next active edge
	cursamp = 0;
	do {
		// Initializations
		fnd_edge = 0;

		// Copy 'current' values to 'last' values
		LastTime     = Time;
		LastClkValue = ClkValue;
		if (inputs)
		for (i = 0; i < InNumSigs; i++)
			LastInSigValue[i] = InSigValue[i];
		for (i = 0; i < OutNumSigs; i++)
			LastOutSigValue[i] = OutSigValue[i];
		for (i = 0; i < MonSigs.nSigs; i++)
			LastMonSigValue[i] = MonSigValue[i];

		// Get a new line from wavefile
		if (ss_readrow(WFile, &Time, Values) <= 0)
			return(1);
		Time *= 1e9;  // convert to nanoseconds
		
		// Hack to skip lines with redundant times 
		// (yes, CAzM sometimes puts them in)
		if (Time == LastTime)
			continue;
		DeltaT = Time-LastTime;

		// clock - assumed to be first dependent var column.
		ClkValue = Values[0];
      
		// input signals
		if (inputs)
			for (i = 0; i < InNumSigs; i++)	{
				InSigValue[i] = Values[i+1];
			}

		// output signals
		for (i = 0; i < OutNumSigs; i++) {
			OutSigValue[i] = Values[1+InNumSigs+i];
		}
		// monitored signals
		for (i = 0; i < MonSigs.nSigs; i++) {
			MonSigValue[i] = Values[1+InNumSigs+OutNumSigs+i];
		}

		// If first call, skip straight to second input vector since LastXXX fields are invalid
		if (firstcall) {
			firstcall = 0;
			continue;
		}

		// Check all outputs: if any make transistions, do hold check
		// and update transition time.
		for (i = 0; i < OutNumSigs; i++) { 
			double lastval   = LastOutSigValue[i];
			double val       = OutSigValue    [i];
			EtffData *ed = SigChk[i].edata;
			char lastbit = logic_thresh(ed, lastval);
			char bit = logic_thresh(ed, val);
			double lstedge;  // last active edge for this signal

			switch(OutSigs[i].edgestyle) {
			case EDGE_BOTH:
				lstedge = EdgeTime;
				break;
			case EDGE_RISE:
				lstedge = LastClkRise;
				break;
			case EDGE_FALL:
				lstedge = LastClkFall;
				break;
			}

			if(lastbit == bit)
				continue;

			if(lastbit == '0') {	// rising.
				// interpolate to time when signal voltage
				// crossed threshold.
				double timeL = LastTime + 
				    DeltaT*((ed->Vil-lastval)/(val-lastval));
				if(OutSigs[i].ignore != IGNORE_DELAY
				   && SigChk[i].do_hold
				   && timeL - lstedge < SigChk[i].hold_L-TIMING_EPSILON) {
					SigChk[i].holderr = 1;

					if(g_verbose_err && report)
					    fprintf(stderr, "%.3f Hold violation on "
						    "%s: Rise=%.3f hold_L=%.3f (%f)\n", 
						    lstedge, OutSigs[i].name,
						    timeL, SigChk[i].hold_L,
						    SigChk[i].hold_L - (timeL - lstedge));
				}
			}
			if(lastbit == '1') {	// falling.
				double timeH = LastTime + 
				    DeltaT*((ed->Vih-lastval)/(val-lastval));
				if(OutSigs[i].ignore != IGNORE_DELAY
				   && SigChk[i].do_hold
				   && timeH - lstedge < SigChk[i].hold_H-TIMING_EPSILON) {
					SigChk[i].holderr = 1;

					if(g_verbose_err && report)
					  fprintf(stderr, "%.3f Hold violation on "
						  "%s: Fall=%.3f hold_H=%.3f (%f)\n",
						  lstedge, OutSigs[i].name,
						  timeH, SigChk[i].hold_H,
						  SigChk[i].hold_H - (timeH - lstedge));
				}
			}
			// If signal has become valid, interpolate to get
			// actual time it crossed threshold, and record for use
			// in setup check.
			if(bit == '1') {	// risen from 0 or X to 1.
				double timeH = LastTime + 
				    DeltaT*((ed->Vih-lastval)/(val-lastval));
				SigChk[i].time_last_X = timeH;
			}
	
			if(bit == '0') {	// fallen from 1 or X to 0.
				double timeL = LastTime + 
				    DeltaT*((ed->Vil-lastval)/(val-lastval));
				SigChk[i].time_last_X = timeL;
			}
		}

		// Check clock for the transitions we care about.

		// Stop conditions for different active edges
		if (LastClkValue >= VSampX && VSampX > ClkValue) {
			fnd_edge = EDGE_FALL;
#ifdef DEBUG
			fprintf(stderr, "Time=%lf; Clock Falls\n", Time);
#endif
		}
		if (LastClkValue <= VSampX && VSampX < ClkValue) {
			fnd_edge = EDGE_RISE;
#ifdef DEBUG
			fprintf(stderr, "Time=%lf; Clock Rises\n", Time);
#endif
		}

	} while ( (ActiveEdge & fnd_edge) == 0);

	// A clock transition just occured.
	// Interpolate to get actual time that clock crossed Vdd/2.
	EdgeTime = LastTime + (Time-LastTime)*(VSampX-LastClkValue)
						/(ClkValue-LastClkValue);

	if(fnd_edge == EDGE_RISE)
		LastClkRise = EdgeTime;
	if(fnd_edge == EDGE_FALL)
		LastClkFall = EdgeTime;

#ifdef DEBUG
	   fprintf(stderr, "Time = %g, LastTime = %g, EdgeTime = %g\n",
		   Time, LastTime, EdgeTime);
#endif

	   // Build input vector by sampling each signal value at EdgeTime
	   if (inputs)
		for (i = 0; i < InNumSigs; i++) {
			double lastval, val, interpval;
	  
			// Linearly interpolate signal value
			lastval   = LastInSigValue[i];
			val       = InSigValue    [i];
			interpval = lastval + (val-lastval)*(EdgeTime-LastTime)
							/(Time-LastTime);

			if((InSigs[i].edgestyle & fnd_edge) == 0)
				invec[i] = '-';
			else
				invec[i] = logic_thresh(InSigChk[i].edata, interpval);
		}
	   // monitor vector is done about the same way.
	   for (i = 0; i < MonSigs.nSigs; i++) {
		   double lastval, val, interpval;
	  
		   // Linearly interpolate signal value
		   lastval   = LastMonSigValue[i];
		   val       = MonSigValue    [i];
		   interpval = lastval + (val-lastval)*(EdgeTime-LastTime)
			   /(Time-LastTime);
		   
		   monvec[i] = logic_thresh(MonSigChk[i].edata, interpval);
	   }


  // Build output vector by sampling each signal value at EdgeTime.
  // Do setup check by comparing edge time with last transition time.
  // Set flag in error vector if setup fails or hold failed above.

	min_slack = DBL_MAX;
	min_slack_sig = -1;
	for (i = 0; i < OutNumSigs; i++) {
		double lastval, val, interpval;
      
		// skip sampling and timing checks if signal is not
		// active on this edge.
		if ((OutSigs[i].edgestyle & fnd_edge) == 0) {
			outvec[i] = '-';
			delayvec[i] = ' '; // initialize
			continue;
		}

		// Linearly interpolate signal value
		lastval   = LastOutSigValue[i];
		val       = OutSigValue    [i];
		interpval = lastval + (val-lastval)*(EdgeTime-LastTime)
						/(Time-LastTime);

		outvec[i] = logic_thresh(SigChk[i].edata, interpval);
		delayvec[i] = ' '; // initialize

		if (OutSigs[i].ignore == IGNORE_DELAY)
			continue;

		if(SigChk[i].holderr)
			delayvec[i] = 'h';
		SigChk[i].holderr = 0;


		// do setup check
		switch(outvec[i]) {
		case '0':
			sig_slack = (EdgeTime - SigChk[i].time_last_X)
				- SigChk[i].setup_L;
			if(sig_slack < min_slack) {
				min_slack = sig_slack;
				min_slack_sig = i;
			}
			
			if(sig_slack < 0) {
				delayvec[i] = 's';
				if(g_verbose_err && report)
					fprintf(stderr, "%.3f Setup violation on %s: LastX=%.3f setup_L=%.3f delta=%.3f\n",
						EdgeTime, OutSigs[i].name,
						SigChk[i].time_last_X,
						SigChk[i].setup_L, -sig_slack);
			}
			break;
		case '1':
			sig_slack = (EdgeTime - SigChk[i].time_last_X)
				- SigChk[i].setup_H;
			if(sig_slack < min_slack) {
				min_slack = sig_slack;
				min_slack_sig = i;
			}

			if(sig_slack < 0) {
				delayvec[i] = 's';
				if(g_verbose_err && report)
					fprintf(stderr, "%.3f Setup violation on %s: LastX=%.3f setup_H=%.3f delta=%.3f\n",
						EdgeTime, OutSigs[i].name,
						SigChk[i].time_last_X,
						SigChk[i].setup_H, -sig_slack);
			}
			break;
		case 'X':
		case 'x':
			break;
		}
	}

	if(min_slack_sig == -1)
		*slacktime = 0;
	else
		*slacktime = min_slack;

	*time = EdgeTime;
	*edgetype = fnd_edge;
	return(0);
}

///////////////////////////////////////////////////////////////////////////
// Print out usage info
//////////////////////////////////////////////////////////////////////////
static void usage()
{
	fprintf(stderr, "Usage:  %s [options] <infile> <wavefile>\n", progname);
	fprintf(stderr, "        <infile>     is a doirsim .in file\n");
	fprintf(stderr, "        <wavefile>   is the file containing analog waveform data\n");
	fprintf(stderr, "  options:\n");
	fprintf(stderr, "        -C <clktemplate>  Name of template file containing clock specs\n");
	fprintf(stderr, "        -N             waveform data does not include input signals\n");
	fprintf(stderr, "        -W <model_dir> directory for ETFO waveform data files\n");
	fprintf(stderr, "        -c <case>      case for choosing ETFO waveform data files\n");
	fprintf(stderr, "        -m             don't include monitored signals in output\n");
	fprintf(stderr, "        -p <period>    simulation done with specified clock period,");
	fprintf(stderr, "                       instead of default.  Period is in nanoseconds.\n");
	fprintf(stderr, "        -t <type>      format of wavefile; (cazm, hspice, etc.) \n");
	fprintf(stderr, "        -v             verbose messages about timing violations\n");
	exit(3);
}


///////////////////////////////////////////////////////////////////////////
// Main routine
//////////////////////////////////////////////////////////////////////////

int main(int argc, char **argv)
{
	static char invec     [MAX_SIGS];
	static char outvec    [MAX_SIGS];
	static char delayvec  [MAX_SIGS];       
	static char monvec    [MAX_SIGS];       
	double time;
	int   cycle;
	int   eof;
	int c;
	char *infile;
	extern char *optarg;
	extern int optind;
	extern int optopt;
	char *etfo_dir = NULL;
	char *fcase = NULL;
	char *WaveFileName;
	char *wavefile_type = "cazm";
	double slack;
	double min_slack;
	int edge_override = -1;
	char *clock_name = NULL;

	// Flags
	int inputs=1;	 // CAzM waveforms and output files include input signals ?
	int do_monsigs=1; // include monitored signals in output.

	// Initialization
	eof = 0;

#ifdef DEBUG
	g_verbose_err = 1;
	g_debug = 1;
#endif

	// parse arguments
	while((c = getopt(argc, argv, "C:NW:c:e:mp:vxt:")) != -1) {
		switch(c) {
		case 'C':
			clock_name = optarg;
			break;
		case 'N':
			inputs = 0;
			break;
		case 'W':
			etfo_dir = optarg;
			break;
		case 'c':
			fcase = optarg;
			break;
		case 'm':
			do_monsigs = 0;
			break;
		case 'p':
			do_clock_period_override = 1;
			clock_period_override = atof(optarg);
			break;
		case 't':
			wavefile_type = optarg;
			break;
		case 'v':
			g_verbose_err = 1;
			break;
		case 'x':
			g_debug = 1;
			break;
		default:
			fprintf(stderr, "%s: Unrecognized option: -%c\n", progname, optopt);
			usage();
			break;
		}
	}
	if(optind + 2 != argc) {
		fprintf(stderr, "%s: infile and wavefile not specified\n", progname);
		usage();
	}

	infile = argv[optind];
	WaveFileName = argv[optind+1];

	etff_list = new EtffList(etfo_dir, fcase);

	// Parse .in file header 
	parse_infile_header(infile);

	// Read clock waveform information
	ClockInfo *cki;
	if(clock_name) {
		cki = read_clock_file(clock_name);
	} else {
		cki = read_clock_file("Clock");
		if(!cki) {
			cki = read_old_clock_file("Clk", etfo_dir, fcase);
		}
	}
	if(!cki)
		exit(2);
	VSampX = cki->Vsamp;

#ifdef DEBUG
	fprintf(stderr, "ActiveEdge=%c\n", edge_chars[ActiveEdge]);
#endif
	// set up signal sampling and timing information
	build_checkinfo();

	// Open spice waveform file
	if ((WFile = ss_open(WaveFileName, wavefile_type)) == NULL) {
		perror(WaveFileName);
		exit(2);
	}
	if(WFile->ndv < OutNumSigs + InNumSigs) {
		fprintf(stderr, "%s has too few signals for use with %s\n",
			WaveFileName, infile);
		exit(2);
	}
	if(WFile->ivar->type != TIME) {
		fprintf(stderr, "%s is not a the result of a transient analysis\n",
			WaveFileName);
		exit(2);
	}

	print_outfile_header("", inputs, do_monsigs, 0);
	
	// Wait for the first falling edge of Clk, no matter what the active edges are.
	// Our cannonical model specifies that a falling clock 
	// transition occurs shortly after the start of simulation,
	// and that this edge is an "initialization" edge not to be compared
	// against any test vectors.
	// Simply waiting for the first edge is not good enough, 
	// because there is sometimes a rising transition on Clk
	// (when powerup simulation is done ?) during the first fraction of ns.

	int etype;
	ActiveEdge = EDGE_FALL;
	if (parse_wave_vector(invec, outvec, delayvec, monvec, &time, inputs, &etype, 0, &slack))
		eof = 1;
	
	if(edge_override != -1)
		ActiveEdge = edge_override;
	else
		ActiveEdge = InfileActiveEdge;
	
	if(g_debug)
		fprintf(stderr, "Skipped %c edge at %.2f ns\n",
			edge_chars[etype], time);

	// if ActiveEdge is 'both' or 'rise', skip second vector,
	// which should be a rising edge.
	if(!eof && ((ActiveEdge & EDGE_RISE)!=0) ) {
		if (parse_wave_vector(invec, outvec, delayvec, monvec, &time, inputs, &etype, 0, &slack))
			eof = 1;
		if(g_debug)
			fprintf(stderr, "Skipped %c edge at %.2f ns\n",
				edge_chars[etype], time);
	}

	// Build a vector for each active edge and write it out
	min_slack = DBL_MAX;
	cycle = 0;
	while (!eof) {

		// Get next vector from wave file
		if (parse_wave_vector(invec, outvec, delayvec, monvec, &time,
				      inputs, &etype, 1, &slack)) {
			eof = 1;
			break;
		} 
		if(ActiveEdge == EDGE_BOTH) {
			if(etype == EDGE_FALL)
				cycle++;
		} else {
			cycle++;
		}
      
		// Write out vector followed by delay information

		print_vector(stdout, invec, outvec, 
			     do_monsigs ? monvec : NULL, inputs, 0,
			     "  %c %-4d#%7.2fns slack= %5.2f",
			     tolower(edge_chars[etype]), cycle, time, slack);

		print_err_vector(stdout, NULL, delayvec, inputs, 0, 0, "");
		if(slack < min_slack)
			min_slack = slack;
	}

	if(cycle > 0) {
		printf("# minimum slack = %.2f\n", min_slack);
	}
		
	ss_close(WFile);
	exit(0);
}
