///////////////////////////////////////////////////////////////////////////
//
// vec.C - utility routines for dealing with test vectors
//	mostly common code factored out of other utilities
//
// Steve Tell Dec 9, 1994
//
//////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <ctype.h>
#include "inparse.h"
#include "vec.h"

///////////////////////////////////////////////////////////////////////////
// print an underlined string
///////////////////////////////////////////////////////////////////////////
void
fputs_ul(char *s, FILE *fp)
{
	char c;
	while(c = *s++) {
		putc(c, fp);
		putc('\b', fp);
		putc('_', fp);
	}
}

///////////////////////////////////////////////////////////////////////////
// print a bold string
///////////////////////////////////////////////////////////////////////////
void
fputs_bold(char *s, FILE *fp)
{
	char c;
	while(c = *s++) {
		putc(c, fp);
		putc('\b', fp);
		putc(c, fp);
	}
}

///////////////////////////////////////////////////////////////////////////
// print character of a highlighted bit vector
//	returns:
//		1 if any error was indicated.
///////////////////////////////////////////////////////////////////////////
int
hl_putc(char datc, char errc, FILE *fp)
{
	int rc = 0;
	putc(datc, fp);
	switch(errc) {
		case ' ':
			break;
		case 's':
		case 'h':
			putc('\b', fp);
			putc(datc, fp);
			rc = 1;
			break;
		case '^':
		default:
			putc('\b', fp);
			putc('_', fp);
			rc = 1;
			break;
	}
	return rc;
}

///////////////////////////////////////////////////////////////////////////
// print a single field
// inputs:
//	fld	pointer to field structure
//	vec	pointer into vector to first bit of field, NULL if none
//	evec	pointer into error vector to first bit of field, NULL if none
//	dobase	if 1, do wide fields as per FORMAT=, else always binary
// 
//	if both vec and evec are non-NULL, we do highlighting
//	vec and evec can't both be NULL.
//
// returns:
//	1 if any error was detected in a field, 0 if not
//	(always 0 if no error vector supplied)
///////////////////////////////////////////////////////////////////////////
static int print_field(FILE *fp, Field *fld, char *vec, char *evec, int dobase)
{
	int i;

	int d;	// digit in progress, takes on value 0-7 or 0-15
	int n;	// iterates over bits in vec
	int dp;	// "digit pointer", index into buf
	int nd; // number of digits
	int x;	// mask of 'x' bits in current digit
	int emask;	// mask of data-error bits in current digit
	int tmask;	// mask of timing-error bits in current digit
	int maxdig;	// maximum value of current digit
	int l2base;  // log base 2 of base
	char conv;
	int rc = 0;

	int style;
#define NORM 1
#define ERR 2
#define HILITE 3

	unsigned long long dx;
	unsigned long long dmval;
	unsigned long long decval;
	unsigned long long demask;
	unsigned long long dtmask;

	char buf[MAX_FIELD_SIGS];
	static char hexd[] = "0123456789abcdef";

	if(vec && evec) {
		style = HILITE;
	} else if(vec && !evec) {
		style = NORM;
	} else if(!vec && evec) {
		style = ERR;
	} else 
		abort();

	if(dobase)
		conv = fld->conv;
	else
		conv = 'b';

	/* if field is "n/a" on this cycle, just print dashes */
	if(style != ERR && vec[0] == '-') {
		int w;
		w = dobase ? fld->width : fld->nsigs;
		for(i = 0; i < w; i++)
			putc('-', fp);
		return 0;
	}

	switch(conv) {
	case 'b':
		for (i = 0; i < fld->nsigs; i++)
		switch(style) {
		case NORM:
			putc(vec[i], fp);
			break;
		case ERR:
			putc(evec[i], fp);
			break;
		case HILITE:
			rc |= hl_putc(vec[i], evec[i], fp);
			break;
		}
		break;

	case 'h':
	case 'o':
		if(fld->conv == 'h')
			l2base = 4;
		else
			l2base = 3;
			

		dp = MAX_FIELD_SIGS-1;
		buf[--dp] = '\0';

		d = 0;
		i = 0;
		nd = 0;
		x = 0;
		emask = 0;
		tmask = 0;
		maxdig = 0;
		for(n = fld->nsigs-1; n >= 0; n--) {
			if(style & NORM) {
				if(vec[n] == 'X') {
					x |= (1 << i);
				} else {
					d |= ((vec[n] & 1) << i);
				}
			}
			if(style & ERR) {
				if(evec[n] == '^') {
					emask |= (1<<i);
					rc |= 1;
				}
				if(evec[n] == 's') {
					tmask |= (1<<i);
					rc |= 1;
				}
			}
			maxdig |= (1 << i);
			i++;
			if(i == l2base) {
				if(style & NORM) {
					if(x == 0)
						buf[--dp] = hexd[d];
					else if(x == maxdig)
						buf[--dp] = 'x';
					else 
						buf[--dp] = 'X';
				}
				if(style == ERR) {
					if(emask)
						buf[--dp] = '^';
					else if(tmask == maxdig)
						buf[--dp] = 's';
					else if(tmask)
						buf[--dp] = 'S';
					else
						buf[--dp] = ' ';
				}

				maxdig = 0;
				x = 0;
				i = 0;
				d = 0;
				tmask = 0;
				emask = 0;
				nd++;
			}
		}
		if(i) {	// do most significant digit if partial
			if(style & NORM) {
				if(x == 0)
					buf[--dp] = hexd[d];
				else if(x == maxdig)
					buf[--dp] = 'x';
				else 
					buf[--dp] = 'X';
				nd++;
			}
			if(style == ERR) {
				if(emask)
					buf[--dp] = '^';
				else if(tmask && tmask == maxdig)
					buf[--dp] = 's';
				else if(tmask)
					buf[--dp] = 'S';
				else
					buf[--dp] = ' ';
			}
		}

		// change leading zeros to spaces if desired
		if(!fld->leadzero) {
			for(i = dp; buf[i] == '0' && buf[i+1]; i++)
				buf[i] = ' ';
		}

		if(style == HILITE) {
			if(emask)
				fputs_ul(&buf[dp], fp);
			else if(tmask)
				fputs_bold(&buf[dp], fp);
			else
				fputs(&buf[dp], fp);
			
		} else {
			fputs(&buf[dp], fp);
		}
		break;

	case 'd':
	case 'u':
		dx = 0;
		dmval = 0;
		decval = 0;
		demask = 0;
		dtmask = 0;
		x = 0;
		for(i = 0; i < fld->nsigs; i++) {
			decval <<= 1;
			dmval <<= 1;
			dx <<= 1;
			dmval |= 1;
			if(style & NORM) {
				if(vec[i] == 'X')
					dx |= 1;
				else
					decval |= (vec[i] & 1);
			}
			if(style & ERR) {
				if(evec[i] == '^') {
					demask |= 1<<i;
					rc |= 1;
				}
				if(evec[i] == 's') {
					dtmask |= 1<<i;
					rc |= 1;
				}
			}
		}

		dp = MAX_FIELD_SIGS-1;
		buf[--dp] = 0;
		nd = 0;

		if(style & NORM) {
			if(dx) {
				if(dx == dmval)
					buf[--dp] = 'x';
				else
					buf[--dp] = 'X';
				nd++;
			} else {
				do {
					buf[--dp] = (decval % 10) + '0';
					decval /= 10;
					nd++;
				} while(decval > 0);
			} 
		}
		if(style == ERR) {
			char c;
			if(demask)
				c = '^';
			else if(dtmask)
				c = 's';
			else
				c = ' ';
			for(i = 0; i < fld->width; i++)
				buf[--dp] = c;
		} else {
			for(; nd < fld->width; nd++) {
				if(fld->leadzero) {
					if(dx == 0)
						buf[--dp] = '0';
					else if(dx == dmval)
						buf[--dp] = 'x';
					else
						buf[--dp] = 'X';
				} else {
					buf[--dp] = ' ';
				}
			}
		}

		if(style == HILITE) {
			if(demask)
				fputs_ul(&buf[dp], fp);
			else if(dtmask)
				fputs_bold(&buf[dp], fp);
			else
				fputs(&buf[dp], fp);
		} else {
			fputs(&buf[dp], fp);
		}
		break;
	}
	return rc;
}

///////////////////////////////////////////////////////////////////////////
// Print a signal vector
// 
// On entry:
//   invec,  outvec, monvec  = in, out, monitor, vectors to print
//                     if invec or monvec are NULL, equivalent number
//                      of spaces are printed instead.		     
//   inputs	     = flag indicates inputs are present.
//   suffixfmt       = printf-like format string for suffix after vector
//////////////////////////////////////////////////////////////////////////
void print_vector(FILE *fp, char *invec, char *outvec, char *monvec,
			 int inputs, int dobase, char *suffixfmt, ...)
{
	int i, j;

	if (inputs) {
		if(!invec) {	// get this case out of the way
			for (i = 0; i < InNumFields; i++) {
				if (i > 0)
					putc(' ', fp);
				for (j = 0; j < InFields[i].width; j++)
					putc(' ', fp);
			}
		}

		for (i = 0; i < InNumFields; i++) {
			if (i > 0)
				putc(' ', fp);
			print_field(fp, &InFields[i], &invec[InFields[i].firstsig], NULL, dobase);
		}
		putc(' ', fp);
	}
// Outputs
	for (i = 0; i < OutNumFields; i++) {
		if (i > 0 || inputs)
			putc(' ', fp);
		print_field(fp, &OutFields[i], &outvec[OutFields[i].firstsig], NULL, dobase);
	}

// Monitored sigs
	if(monvec && MonSigs.nFields) {
		putc(' ', fp);
		putc(' ', fp);
		for (i = 0; i < MonSigs.nFields; i++) {
			if (i > 0)
				putc(' ', fp);
			if(monvec) {
				print_field(fp, &MonSigs.Fields[i], 
					    &monvec[MonSigs.Fields[i].firstsig],
					    NULL, dobase);
			} else {
				for (j = 0; j < MonSigs.Fields[i].width; j++)
					putc(' ', fp);
			}
		}
	}

	va_list ap;
	va_start(ap, suffixfmt);
	vfprintf(fp, suffixfmt, ap);
	va_end(ap);

	putc('\n', fp);
}

///////////////////////////////////////////////////////////////////////////
// Print an error vector
// 
// On entry:
//   invec,  outvec  = in, out vectors to print
//                     if invec is NULL, equivalent number of spaces are
//                     printed instead.		     
//   inputs	     = flag indicates inputs are present.
//   domonitor       = flag indicates to print spaces for monitored signals
//   suffixfmt       = printf-like format string for suffix after vector
//////////////////////////////////////////////////////////////////////////
void print_err_vector(FILE *fp, char *invec, char *outvec,
			 int inputs, int domonitor, int dobase, char *suffixfmt, ...)
{
	int i, j;
	int w;

	if (inputs) {
		if(!invec) {	// get this case out of the way
			for (i = 0; i < InNumFields; i++) {
				if (i > 0)
					putc(' ', fp);

				w = dobase?InFields[i].width:InFields[i].nsigs;
				for (j = 0; j < w; j++)
					putc(' ', fp);
			}
		} else {
			for (i = 0; i < InNumFields; i++) {
				if (i > 0)
					putc(' ', fp);
				print_field(fp, &InFields[i], NULL, &invec[InFields[i].firstsig], dobase);
			}
		}
		putc(' ', fp);
	}

	for (i = 0; i < OutNumFields; i++) {
		if (i > 0 || inputs)
			putc(' ', fp);
		print_field(fp, &OutFields[i], NULL, &outvec[OutFields[i].firstsig], dobase);
	}
	// monitored signals never have errors, just print the spaces
	if(domonitor && MonSigs.nFields) {
		putc(' ', fp);
		putc(' ', fp);
		for (i = 0; i < MonSigs.nFields; i++) {
			if (i > 0)
				putc(' ', fp);
			for (j = 0; j < MonSigs.Fields[i].width; j++)
				putc(' ', fp);
		}
	}
	va_list ap;
	va_start(ap, suffixfmt);
	vfprintf(fp, suffixfmt, ap);
	va_end(ap);

	putc('\n', fp);
}

///////////////////////////////////////////////////////////////////////////
// Print a highlighted vector:
//	Data errors underlined	(inverse in xterm with default termcap)
//	Setup errors in bold
// 
// On entry:
//   invec,  outvec  = in, out vectors to print
//                     if invec is NULL, equivilent number of spaces are
//                     printed instead.		     
//   inerrvec, outerrvec =
//			error vectors to control display style.
//			'^' for data errors, s for timing.
//
//   inputs	     = flag indicates inputs are present.
//   suffixfmt       = printf-like format string for suffix after vector
//////////////////////////////////////////////////////////////////////////
void print_hl_vector(FILE *fp, char *invec, char *outvec, char *monvec,
		     char *inerrvec, char *outerrvec,
			 int inputs, int dobase, char *suffixfmt, ...)
{
	int i, j;
	int w;
	int err = 0;	// set to 1 if any error indicated on line.

	if (inputs) {
		if(!invec) {	// get this case out of the way
			for (i = 0; i < InNumFields; i++) {
				if (i > 0)
					putc(' ', fp);

				w = dobase?InFields[i].width:InFields[i].nsigs;
				for (j = 0; j < w; j++)
					putc(' ', fp);
			}
		}

		for (i = 0; i < InNumFields; i++) {
			if (i > 0)
				putc(' ', fp);
			err |= print_field(fp, &InFields[i], 
				    &invec[InFields[i].firstsig],
				    &inerrvec[InFields[i].firstsig], dobase);
		}
		putc(' ', fp);
	}

	for (i = 0; i < OutNumFields; i++) {
		if (i > 0 || inputs)
			putc(' ', fp);
		err |= print_field(fp, &OutFields[i],
			    &outvec[OutFields[i].firstsig],
			    &outerrvec[OutFields[i].firstsig], dobase);
	}

	if(monvec && MonSigs.nFields) {
		putchar(' ');
		putchar(' ');
		for (i = 0; i < MonSigs.nFields; i++) {
			if (i > 0)
				putc(' ', fp);
			if(monvec) {
				(void) print_field(fp, &MonSigs.Fields[i],
					   &monvec[MonSigs.Fields[i].firstsig],
					   NULL, dobase);
			} else {
				for (j = 0; j < MonSigs.Fields[i].width; j++)
					putc(' ', fp);
			}

		}
	}

	char buf[1024];

	va_list ap;
	va_start(ap, suffixfmt);
	vsprintf(buf, suffixfmt, ap);
	va_end(ap);

	if(err) {
		fputs_ul(buf, fp);
	} else {
		fputs(buf, fp);
	}

	putc('\n', fp);
}


///////////////////////////////////////////////////////////////////////////
// get pointer to column label string for jth digit of ith field
// helper routine for print_outfile_header
//	
// Inputs:
//	flds is either InFields or OutFields
//	sigs is either InSigs or OutSigs
//	fn is field number
//	dn is digit number within that field,
//		counted from the left starting with zero.
// 
///////////////////////////////////////////////////////////////////////////
static char *get_col_hdr(Field *flds, Sig *sigs, int fn, int dn, int dobase)
{
	int l2base;
	char conv;

	if(dobase)
		conv = flds[fn].conv;
	else
		conv = 'b';

	switch(conv) {
	case 'b':
		return sigs[flds[fn].firstsig + dn].name;
	case 'd':
	case 'u':
		return flds[fn].name;
		break;
	case 'h':
		l2base = 4;
		break;
	case 'o':
		l2base = 3;
		break;
	default:
		abort();
	}
	// if we get here, its either hex or octal

	// digit number counted from the right, starting at 1
	int dr = flds[fn].width - dn;

	// bit number of leftmost bit of the digit, 
	// counted from the right starting with 1
	int br = dr*l2base;

	if(br > flds[fn].nsigs)
		br = flds[fn].nsigs;

	return sigs[flds[fn].firstsig + flds[fn].nsigs - br].name;
}

///////////////////////////////////////////////////////////////////////////
// Routine to generate a header with the signal names written vertically 
// followed by a row of '===='s
//
// Translated from the awk 6/23/94 by Steve Molnar
//
// The name printed for each format is as follows:
//	'b':		simply the signal name
//	'd', 'u':	the un-iterated field name
//	'h', 'o':	the name of the signal corresponding to the
//			leftmost bit in the field.
//
// args:
//	msg: printed at the top
//	inputs: are inputs present
//	dobase: 1 if groups should be displayed as per the FORMAT= attribute
//
//////////////////////////////////////////////////////////////////////////
void print_outfile_header(char *msg, int inputs, int domonitor, int dobase)
{
	int    len, maxlen;
	int    f, d, i, w;
	char *cp;

	// Print first line
	printf("%s", msg);
 
	// Figure out the length of the longest label string
	maxlen = 0;
	if (inputs)
		for (f = 0; f < InNumFields; f++) {
			w = dobase ? InFields[f].width : InFields[f].nsigs;
			for (d = 0; d < w; d++) {
				cp = get_col_hdr(InFields, InSigs, f, d, dobase);
				len = strlen(cp);
				if (len > maxlen)
					maxlen = len;
			}
		}

	for (f = 0; f < OutNumFields; f++) {
		w = dobase ? OutFields[f].width : OutFields[f].nsigs;
		for (d = 0; d < w; d++) {
			cp = get_col_hdr(OutFields, OutSigs, f, d, dobase);
			len = strlen(cp);
			if (len > maxlen)
				maxlen = len;
		}
	}
	if(domonitor) {
		for (f = 0; f < MonSigs.nFields; f++) {
			w = dobase ? MonSigs.Fields[f].width : MonSigs.Fields[f].nsigs;
			for (d = 0; d < w; d++) {
				cp = get_col_hdr(MonSigs.Fields, MonSigs.Sigs, f, d, dobase);
				len = strlen(cp);
				if (len > maxlen)
					maxlen = len;
			}
		}
	}

	// Print label lines
	putchar('\n');
	for (i = 0; i < maxlen; i++) {
		// Inputs
		if (inputs) {
			for (f = 0; f < InNumFields; f++) {
				w = dobase ? InFields[f].width : InFields[f].nsigs;
				for (d = 0; d < w; d++) {
					cp = get_col_hdr(InFields, InSigs, f, d, dobase);
					if (strlen(cp) > i)
						putchar(cp[i]);
					else
						putchar(' ');
				}
				putchar(' ');
					
			}
			putchar(' ');
		}

		// Outputs
		for (f = 0; f < OutNumFields; f++) {
			w = dobase ? OutFields[f].width : OutFields[f].nsigs;
			for (d = 0; d < w; d++) {
				cp = get_col_hdr(OutFields, OutSigs, f, d, dobase);
				if (strlen(cp) > i)
					putchar(cp[i]);
				else
					putchar(' ');
			}
			if (f < OutNumFields-1)
				putchar(' ');
		}
		// monitored signals
		if(domonitor && MonSigs.nFields) {
			putchar(' ');
			putchar(' ');
		}
		for (f = 0; domonitor && f < MonSigs.nFields; f++) {
			w = dobase ? MonSigs.Fields[f].width : MonSigs.Fields[f].nsigs;
			for (d = 0; d < w; d++) {
				cp = get_col_hdr(MonSigs.Fields, MonSigs.Sigs, f, d, dobase);
				if (strlen(cp) > i)
					putchar(cp[i]);
				else
					putchar(' ');
			}
			if (f < MonSigs.nFields-1)
				putchar(' ');
		}


		putchar('\n');
	}
  
	// Print divider line
	if (inputs) {
		for (f = 0; f < InNumFields; f++) {
			w = dobase ? InFields[f].width : InFields[f].nsigs;
			for (d = 0; d < w; d++)
				putchar('=');
			if(f < InNumFields-1)
				putchar('=');
		}
		putchar(' ');
		putchar(' ');
	}
	for (f = 0; f < OutNumFields; f++) {
		w = dobase ? OutFields[f].width : OutFields[f].nsigs;
		for (d = 0; d < w; d++)
			putchar('=');
		if (f < OutNumFields-1)
			putchar('=');
	}

	if(domonitor && MonSigs.nFields) {
		putchar(' ');
		putchar(' ');
	}

	for (f = 0; domonitor && f < MonSigs.nFields; f++) {
		w = dobase ? MonSigs.Fields[f].width : MonSigs.Fields[f].nsigs;
		for (d = 0; d < w; d++)
			putchar('=');
		if (f < MonSigs.nFields-1)
			putchar('=');
	}

	putchar('\n');
}
