
#
# require 'syscall.ph'
# syscall.ph is missing on our system, so we declare the one we need
# here.  Fortunately sys_exit seems to be the same number on most unixes.
# This is all a nasty hack because somehow perl's normal exit()
# is failing to pass the return code back to unix. (on some perl's anyway)
#
sub SYS_exit {1;}

package Text::EP3::Spice;

sub myexit { my($rc) = @_; syscall &::SYS_exit, $rc; }

use strict;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);
#use AutoLoader;
use Carp;
use Exporter;

use Data::Dumper;
use FileHandle;

#@ISA = qw(Exporter AutoLoader);
@ISA = qw(Exporter);
# ************  IMPORTANT FOR ALL EP3 MODULES ***********
# Tell EP3 to inherit methods from this module ...
# This is necessary to let EP3 know about the methods ...
push (@Text::EP3::ISA, qw(Text::EP3::Spice));
# *******************************************************
# Set up default values

@EXPORT = qw(
);
$VERSION = '1.00';

use vars qw(%etfd);

sub clock;
sub ladder;
sub trans;
sub wfm;
1;

#__END__

#
# @wfm name
#	for compatibilty with cazmpp, load the named waveform file and
#	insert it.
#
sub wfm {
	my $self = shift;  # EP3 instance
 	my(@input_string) = @_;
	my($inline);
 	$inline = $input_string[0];
	my($directive, $name) = split(' ',$inline);
	my($fname, $dir, $d);

	if(!defined($self->{SimCase})) {
		print STDERR "spicepp \@wfm: no case specified on command line\n";
		myexit(1);
	}
	$fname = $name . '.' . $self->{SimCase};

	foreach $d (@{$self->{WavePath}}) {
		if( -f $d . '/' . $fname) {
			$dir = $d;
			last;
		}
	}
	if(!$dir) {
		print STDERR "File not found for wfm $fname; wavefile path is:\n";
		print STDERR " ", join(':', @{$self->{WavePath}}), "\n";
#		print STDERR Dumper($self->{WavePath}), "\n";
		$self->{Status} = 1;
		&myexit(1);
	}

	$self->ep3_process($dir . '/'. $fname);
}
#
# @ladder instname inputnode outputnode rseg cseg nsegs
#
# Generate RC ladder with nsegs segments, and the specified
# resistance and capacitance in each segment.
# 
sub ladder {
	my $self = shift;  # EP3 instance
 	my(@input_string) = @_;
	my($inline, $i);
 	$inline = $input_string[0];
	my($directive, $inst, $innode, $outnode, $rseg, $cseg, $n) = split(' ',$inline);
	
	printf "* ladder name=%s innode=%s outnode=%s r=%s c=%s n=%d\n",
		$inst, $innode, $outnode, $rseg, $cseg, $n;
	
	printf "R${inst}0 $innode S${inst}0 %s\n", $rseg;
	printf "C${inst}0 S${inst}0 GND %s\n", $cseg;
	for($i = 1; $i < $n-1; $i++) {
		printf "R${inst}%d S${inst}%d S${inst}%d %s\n",
			$i, $i-1, $i, $rseg;
		printf "C${inst}%d S${inst}%d GND %s\n", $i, $i, $cseg;
	}
	printf "R${inst}%d S${inst}%d $outnode %s\n",
		$n-1, $n-2, $rseg;
	printf "C${inst}%d $outnode GND %s\n", $n-1, $cseg;
}


#
# @xladder in0,in1,... out0,out1,... [com0,com1,...] subckt N [parameter=value ... ]
#
# Generate generic ladder of N subcircuit instances.
# first two arguments and optional third argument are lists of signals.
# 
# 
use vars qw( $xladder_base );

sub xladder {
	my $self = shift;  # EP3 instance
 	my(@input_string) = @_;
	my($inline, $i, $j, $nsigs);
 	$inline = $input_string[0];
	my($inlist, $outlist, $comlist, $subckt, $n, $base);
	my(@ins, @outs, @coms);
	
	my(@args) = split(' ',$inline);
	shift(@args);  # "@xladder"
	$inlist = shift(@args);
	@ins = split(',', $inlist);
	$outlist = shift(@args);
	@outs = split(',', $outlist);
	if($#args >= 2 && $args[2] !~ /=/) {
		$comlist = shift(@args);
		@coms = split(',', $comlist);
	}
	$subckt = shift(@args);
	$n = shift(@args);
	if(!defined($xladder_base)) {
		$xladder_base = 'A';		
	} else {
		$xladder_base++;
	}
	$base = $xladder_base;

	printf "* xladder $base inlist=%s outlist=%s comlist=%s subckt=%s n=%d params=%s\n",
		join(',', @ins), join(',', @outs), join(',', @coms), 
		$subckt, $n, join(' ', @args);	
	if($#ins != $#outs) {
		print STDERR "\@xladder $subckt: different number of ins and outs.\n";
		&myexit(2);
	}
	if($n < 2) {
		print STDERR "\@xladder $subckt: ladder length must be >= 2.\n";
		&myexit(2);
	}
	
	$nsigs = $#ins + 1;
	
	# print first one
	print "x${base}0 ";
	print join(' ', @ins);
	for($j = 0; $j < $nsigs; $j++) {
		printf " L${base}0_%d", $j;
	}
	if($#coms >= 0) { print " ", join(' ', @coms); }
	print " $subckt";
	if($#args >= 0) { print " ", join(' ', @args); }
	print "\n";

	# do intermediate ones
	for($i = 1; $i < $n-1; $i++) {
		printf "x${base}%d", $i;
		for($j = 0; $j < $nsigs; $j++) {
			printf " L${base}%d_%d", $i-1, $j;
		}
		for($j = 0; $j < $nsigs; $j++) {
			printf " L${base}%d_%d", $i, $j;
		}

		if($#coms >= 0) { print " ", join(' ', @coms); }
		print " $subckt";
		if($#args >= 0) { print " ", join(' ', @args); }
		print "\n";

	}
	
	# do last one
	printf "x${base}%d", $n-1;
	for($j = 0; $j < $nsigs; $j++) {
		printf " L${base}%d_%d", $n-2, $j;
	}
	print " ";
	print join(' ', @outs);
	if($#coms >= 0) { print " ", join(' ', @coms); }
	print " $subckt";
	if($#args >= 0) { print " ", join(' ', @args); }
	print "\n";
}


# 
# @trans increment duration
#	duration can either be a time, or a number with a "clk" suffix
#	to indicate a number of clock periods.
#
sub trans {
	my $self = shift;
	my(@input_string) = @_;
	my($inline);
	my($incr, $directive, $dur, $dnum, $tlength);
	my($monte);

	$inline = $input_string[0];
	($directive, $incr, $dur) = split(' ', $inline);

	if(defined $self->{Define_List}{'_monte_carlo_'}) {
		$monte = $self->{Define_List}{'_monte_carlo_'};
	} else {
		$monte = 0;
	}
	if($dur =~ m/(^[0-9.e-]+)clk$/) {
		if(!defined($self->{PeriodNum})) {
			print STDERR "spicepp \@trans: no default clock period specified on command line\n";
			&myexit(1);
		}
		$tlength = ($1 * $self->{PeriodNum}) . $self->{PeriodSuffix};
		&hspice_write_transient($tlength, $incr, $monte);
	} else {
		&hspice_write_transient($dur, $incr, $monte);
	}
}

#
# @clock plus-node minus-node clock-name [period]
#   Generate a PWL voltage source for a clock waveform.
#   Clock-name is used to determine a file from which to get the
#   waveform fragments.
#   period is used to adjust the period of the generated waveform.
#   If no period is specfied, the command-line default period is used.
#
sub clock {
	my $self = shift;  # EP3 instance
 	my(@input_string) = @_;
	my($inline);
	my($ct, $delay);
 	$inline = $input_string[0];
	my($directive, $pnode, $nnode, $name, $period, $delay) = split(' ',$inline);

	printf "* clockdirective n=%s p=%s name=%s period=%f delay=%f\n",
		$pnode, $nnode, $name, $period, $delay;

	if(!defined($self->{SimCase})) {
		print STDERR "spicepp \@clock: case specified on command line\n";
		&myexit(1);
	}
	$ct = $name . '.' . $self->{SimCase};
	if(!defined($etfd{$ct})) {
		$self->read_wave($ct);
	}
	if(!$period) {
		if(!defined($self->{Period})) {
			print STDERR "spicepp \@clock: no default clock period specified on command line\n";
			&myexit(1);
		}
		$period = $self->{Period}*1e9;
	}
	if(!$delay) {
		$delay = 0;
	}

	print STDERR "    Using waveform \"$name\" for clock $name\n";

	my($d);
	$d = $etfd{$ct};

	if(!&emit_etfo($pnode, $nnode, $period, 'B', 
		       1, $delay, $d, '01', '1', 1)) {

		$self->{Status} = 1;
	}
}

#
# @wavebit Pos Neg {bits} ETFO=fanout+delay,edge,wavename
#
# This replaces cazmpp's "wave bit ETFO" syntax.
#
sub wavebit {
	my $self = shift;
	my(@input_string) = @_;
	my($inline);
	my($directive, $posnode, $negnode, $bits, $rest, $bit0);
	my($ct);

	$inline = $input_string[0];
	($directive, $posnode, $negnode, $bits, $rest) = split(/[ \t{}]+/, $inline);
	if($rest !~ m/^ETFO=/) {
		print STDERR "syntax error in \@wavebit: no ETFO= found\n";
		$self->{Status} = 1;
		return;
	}
	my($etfo,$fanout,$delay,$edge,$wavetype) = split(/[,+= \n]+/, $rest);
	if(!$edge) {
		$edge = 'F';
	}
	$edge =~ tr/a-z/A-Z/;
	if(!$wavetype) {
		$wavetype = 'etff';
	}
	if(!defined($self->{SimCase})) {
		print STDERR "spicepp \@wavebit: case specified on command line\n";
		&myexit(1);
	}
	$ct = $wavetype . '.' . $self->{SimCase};
	if(!defined($etfd{$ct})) {
		$self->read_wave($ct);
	}

	print STDERR "    Using waveform \"$wavetype\" for $posnode\n";

	my($d);
	$d = $etfd{$ct};
	$bit0 = substr($bits, 0, 1);
	if($edge eq 'B') {
		$bits = $bit0 . $bit0 . $bits;
	} else {
		$bits = $bit0 . $bits;
	}

	if(!defined($self->{Period})) {
		print STDERR "spicepp \@wavebit: no default clock period specified on command line\n";
		&myexit(1);
	}
	if(!&emit_etfo($posnode, $negnode, ($self->{Period}*1e9), $edge, 
		       $fanout, $delay, $d, $bits, $bit0, 0)) {

		$self->{Status} = 1;
	}
}


#
# Helper routines for emit_etfo.  ensure that it doesn't try to to
# backwards in time in the presence of bogus template files
#
use vars qw( $etfo_last_time $etfo_current_name $etfo_first);
sub point_init
{ 
	my($n) = @_;
	#print "* point_init name=$n\n";
	$etfo_current_name = $n;
	$etfo_last_time = 0.0;
	$etfo_first = 1;
}
sub point_emit
{
	my($t, $v) = @_;
	my $printit = 0;
	if($t < $etfo_last_time) {
		print STDERR "spicepp: trying back up time in a PWL for $etfo_current_name; time=$t v=$v\n";
		print STDERR "perhaps the templates are inconsistent?\n";
		exit 1;
	}
	if($etfo_first) {
		$printit = 1;
	} else {
		my ($tstr, $ltstr);
		$tstr = sprintf "%4.3f", $t;
		$ltstr = sprintf "%4.3f", $etfo_last_time;
		if($tstr+0 > $ltstr+0) {
			$printit = 1;
		}
	}
	if($printit) {
		printf "%4.3fn %4.3f ", $t, $v;
	} else {
		#printf "\n* time increment below print resolution; point skipped: %f %f\n+ ", $t, $v;
	}
	$etfo_first = 0;
	$etfo_last_time = $t;
}


#
# do interpolation and generate PWL for an etfo specification.
# Used for both data waveform generation by @wavebit, and by @clock
# for clock generation.
#
sub emit_etfo {
	my($posnode, $negnode, $pw, $edge, $fanout, $delay, $d, $bits, $initval, $rpt) = @_;
	my($Time, $lastbit, $curbit, $i, $j, $pttime, $vlast, $nline, $tlast);
	my(@bit, $nvd, $nvu, $starti);

	if($$d{pw} > $pw) {
		printf STDERR "Minimum period (%4.3fns) specified in wavefile is greater than command-line\n", $$d{pw};
		printf STDERR "clock cycle time of %4.3fns.\n", $pw;
		return 0;
	}

	@bit = split(' *', $bits);
	&point_init($posnode);
	printf "v$posnode $posnode $negnode pwl ";

	# set initial time.
	if($edge eq 'F') {
		$Time = 0;
	} elsif($edge eq 'R') {
		$Time = $pw/2;
	} elsif($edge eq 'B') {
		$Time = 0;
		$pw /= 2;
	}

	# set our startoff behaviour.  If the initial value is not the
	# same as the first data bit, we need to actually generate that
	# first transition.  Otherwise, we will just set the
	# initial value.
	# (These cases turn out to correspond to
	# data and clock generation, but we're trying to be general)
	$lastbit = $initval;
	if($initval != $bit[0]) {
		$starti = 0;
	} else {
		$starti = 1;
	}		
	# output the initial value. It may seem counter-
	# intuitive to set a 0 bit to vu[0] and vice-versa,
	# but it is right:  an up-going transition starts LOW!
	if ($initval eq '0') {
		&point_emit(0.0, $$d{vu}[0]);
		&point_emit($delay, $$d{vu}[0]);
	}
	if ($initval eq '1') {
		&point_emit(0.0, $$d{vd}[0]);
		&point_emit($delay, $$d{vd}[0]);
	}
		
	$nline = 1;
	$nvd = $#{$$d{vd}};
	$nvu = $#{$$d{vu}};
	for ($i = $starti; $i <= $#bit; $i++) {
		$curbit = $bit[$i];
		if (($curbit == 0) && ($curbit != $lastbit)) {
			# print "\n*i=$i cur=0 last=$lastbit Time=$Time\n";
			
 			for ($j = 0; $j < $nvd+1; $j++) {
				$pttime = $delay + $Time + $$d{tda}[$j]
					+ $fanout
					  * (($$d{tdb}[$j] - $$d{tda}[$j])/10);
				&point_emit($pttime, $$d{vd}[$j]);
				if(++$nline >= 4) {
					print "\n+ ";
					$nline = 0;
				}
			}
		} elsif (($curbit == '1') && ($curbit != $lastbit)) {
			# print "\n*i=$i cur=1 last=$lastbit Time=$Time\n";
			for ($j = 0; $j < $nvu+1; $j++) {
				$pttime = $delay + $Time + $$d{tua}[$j]
					+ $fanout
					  * (($$d{tub}[$j] - $$d{tua}[$j])/10);
				&point_emit($pttime, $$d{vu}[$j]);
				if(++$nline >= 4) {
					print "\n+ ";
					$nline = 0;
				}
			}
		}
		$Time = $Time + $pw;
		$lastbit = $curbit;
	}
	if ($lastbit == 0) {
		$vlast = $$d{vd}[$nvd];
	} elsif ($lastbit == 1) {
		$vlast = $$d{vu}[$nvu];
	}
	# print "* Time=$Time delay=$delay pttime=$pttime\n";
	&point_emit($Time + $delay, $vlast);

	if($rpt) {
		printf " R %4.3fn", $delay;
	}
	print "\n";
	return 1;
}

#
# Read a waveform description file
# side effect: stores data in %etfd hash
# The global %etfd is a hash-of-hashes, used as a hash of record
# structures.  $etfd{$wavekey} is the data for the named
# waveform.  Each waveform has these items associated with it:
#	pw, tz, 
#	t0ph, t0pl, t0r, t0f, 
#	t10ph, t10pl, t10r, t10f, 
#	vd, tda, tdb
#	vu, tua, tub
#
sub read_wave
{
	my $self = shift;
	my($ct) = @_;

	my($i, $j, $state, @Fld, $dir, $ok);
	my($dn_max, $up_max, $fname, $fh);
	local($_);

	$fh = new FileHandle;
	foreach $dir (@{$self->{WavePath}}) {
		$fname = $dir . '/' . $ct;
		if($ok = open($fh, $fname)) {
			last;
		}
	}
	if(!$ok) {
		print STDERR "Wave file not found for $ct; wavefile path is:\n";
		print STDERR " ", join(' ', @{$self->{WavePath}}), "\n";
#		print STDERR Dumper($self->{WavePath}), "\n";
		&myexit(1);
	}

	print STDERR "Loading wave $ct from $fname\n";

	$state = 0;
	line: while (defined($_ = <$fh>)) {
		chop($_);
		$_ =~ s/#.*$//;
		next line if($_ =~ m/^\s*$/);
		@Fld = split(/\s+/, $_);

	# note: parameter times in file are in seconds
	# but we work in nanoseconds.
		if ($Fld[0] eq 'pw') {
			$etfd{$ct}{pw} = 1e9 * $Fld[2];
		} elsif ($Fld[0] eq 'tz') {
			$etfd{$ct}{tz} = 1e9 * $Fld[2];
		} elsif ($Fld[0] eq 't0ph') {
			$etfd{$ct}{t0ph} = 1e9 * $Fld[2];
		} elsif ($Fld[0] eq 't0pl') {
			$etfd{$ct}{t0pl} = 1e9 * $Fld[2];
		} elsif ($Fld[0] eq 't0r') {
			$etfd{$ct}{t0r} = 1e9 * $Fld[2];
		} elsif ($Fld[0] eq 't0f') {
			$etfd{$ct}{t0f} = 1e9 * $Fld[2];
		} elsif ($Fld[0] eq 't10ph') {
			$etfd{$ct}{t10ph} = 1e9 * $Fld[2];
		} elsif ($Fld[0] eq 't10pl') {
			$etfd{$ct}{t10pl} = 1e9 * $Fld[2];
		} elsif ($Fld[0] eq 't10r') {
			$etfd{$ct}{t10r} = 1e9 * $Fld[2];
		} elsif ($Fld[0] eq 't10f') {
			$etfd{$ct}{t10f} = 1e9 * $Fld[2];
		}
	# process the latch table file to build the edge tables:
	# note that here, the time numbers in the file are nanoseconds.
	# backwards compatibility is a bitch.

		if (($#Fld == 0) && ($Fld[0] eq 'down')) {
			$state = 1;
			$i = 0;
		} elsif (($#Fld == 0) && ($Fld[0] eq 'up')) {
			$state = 2;
			$j = 0;
		} elsif ($state == 1) {
			$etfd{$ct}{vd}[$i] = $Fld[0];
			$etfd{$ct}{tda}[$i] = $Fld[1];
			$etfd{$ct}{tdb}[$i] = $Fld[2];
			$i++;
		} elsif ($state == 2) {
			$etfd{$ct}{vu}[$j] = $Fld[0];
			$etfd{$ct}{tua}[$j] = $Fld[1];
			$etfd{$ct}{tub}[$j] = $Fld[2];
			$j++;
		}
	}
	close($fh);

#	print STDERR "%etfd=", Dumper(%etfd), "\n";
#	my($d);
#	$d = $etfd{$ct};
#	print STDERR "d=", Dumper($d), "\n";
#	printf STDERR "pw=%2.3f\n", $$d{pw};
#	printf STDERR "vd[0] = %2.3f\n", $$d{vd}[0];
#	print STDERR "d{vd}=", Dumper($$d{vd}), "\n";
#	printf STDERR "\$#d{vd}=%d\n", $#{$$d{vd}};
#	exit 0;
	
}

#
# convert a number with spice-style suffixes into an ordinary floating-
# point number.  Example: 10n is 10e-9.
#
sub spice2float
{
	my($snum) = @_;
	my($num, $suff);

	if($snum =~ m/^([-+.0-9eE]+)([a-zA-Z]+)/) {
		$num = $1;
		$suff = $2;
		$suff =~ tr/A-Z/a-z/;
		if($suff eq 'f') { return $num * 1e-15; }
		elsif($suff eq 'p') { return $num * 1e-12; }
		elsif($suff eq 'n') { return $num * 1e-9; }
		elsif($suff eq 'u') { return $num * 1e-6; }
		elsif($suff eq 'm') { return $num * 1e-3; }
		elsif($suff eq 'k') { return $num * 1e3; }
		elsif($suff eq 'x') { return $num * 1e6; }
		elsif($suff eq 'meg') { return $num * 1e6; }
		elsif($suff eq 'g') { return $num * 1e9; }
		else {
			print STDERR "unknown scale-factor suffix on number: $snum\n";
			return $num;
		}
	} else {
		return $snum;
	}

}

#
# these hspice-specific output functions should all move to a module,
# so that a different module can be loaded for use with other spice variants.
#
sub hspice_write_transient
{
	my($duration, $incr, $monte) = @_;
	
	print ".tran $incr $duration";
	if($monte) {
		printf " SWEEP MONTE=%d", $monte;
	}
	print "\n";
}
