/* analyze.c --- waveform analysis routines for tcalc system

MODULE
    analyze.c

DESCRIPTION
    This module contains routines for analyzing waveforms and accumulating
    their transition times.

HISTORY
    6/7/88 --- written by Steven Molnar
BUGS

---------------------------------------------------------*/

#include <strings.h>
#include "tcalc.h"
#include "parse.h"

extern SIGNAL *TraceList[];	/* table of signals being traced */
extern int     NumTraces;

extern float   Start;
extern float   Time;
extern float   OldTime;


/************************************************************************
Linear interpolation routines to find crossover points.

Sketch is following:

     |         X V1
     |       / |
     |----/----|thresh
     | /  |    |
  v0 X    |    |
     |    |    |
     t0   t    t1

On entry:
  t0, t1 = times 
  v0, v1 = values
  thresh = crossover value
On exit:
  return crossover time
************************************************************************/
float linear_interp(t0, t1, v0, v1, thresh)
     float t0, t1, v0, v1, thresh;
{
  float result;

  result = t0 + (t1-t0)*(thresh-v0)/(v1-v0);
  if (result < t0 || result > t1)
    error_out("Bad interpolation");
  return(result);
}

  
/************************************************************************
Transition analysis is done by a finite state machine.  States correspond
to a waveform as follows:

  Pre  |    HH                         HH
     __|________                 ____________
    X  |        X               X            X 
   /   |         \ HX          / XH           \ HX
  X    |          X           X                X
 /     |           \ XL      / LX               \ XL
       |            X_______X                    X____
       |Start            LL
--------------------------------------------------------> Time

On entry:
  signal = pointer to signal to analyze
************************************************************************/
void finite_state(s)
     SIGNAL *s;
{
  float val, oval;
  float L, X, H;
  int   oldState;

  val  = s->V;
  oval = s->OldV;
  L    = s->L;
  X    = s->X;
  H    = s->H;

  do 
    {
      oldState = s->state;
      
      switch(s->state)
	{
	case PRE:
	  if (Time >= Start)
	    {
	      oval = s->OldV = val;  /* start things off on the right foot */

	      if (val >= H)
		s->state = HH;
	      else if (val >= X)
		{
		  fprintf(stderr,"Warning: signal <%s> is in transition at start time %g\n",
			  s->name, Time);
		  s->state = XHorHX;
		}
	      else if (val >= L)
		{
		  fprintf(stderr,"Warning: signal <%s> is in transition at start time %g\n",
			  s->name, Time);
		  s->state = XLorLX;
		}
	      else 
		s->state = LL;
	    }
	  break;
	  
	case XHorHX:
	  if (val >= H)
	    {
	      s->RH[s->nRH++] = linear_interp(OldTime, Time, oval, val, H);
	      s->state = HH;
	    }
	  else if (val < X)
	    {
	      s->FX[s->nFX++] = linear_interp(OldTime, Time, oval, val, X);
	      s->state = XL;
	    }
	  break;

	case XLorLX:
	  if (val >= X)
	    {
	      s->RX[s->nRX++] = linear_interp(OldTime, Time, oval, val, X);
	      s->state = XH;
	    }
	  else if (val < L)
	    {
	      s->FL[s->nFL++] = linear_interp(OldTime, Time, oval, val, L);
	      s->state = LL;
	    }
	  break;

	case HH:
	  if (val < H)
	    {
	      s->FH[s->nFH++] = linear_interp(OldTime, Time, oval, val, H);
	      s->state        = HX;
	    }
	  break;
	  
	case HX:
	  if (val < X)
	    {
	      s->FX[s->nFX++] = linear_interp(OldTime, Time, oval, val, X);
	      s->state        = XL;
	    }
	  else if (val >= H)
	    {
	      fprintf(stderr,
		 "Warning:  glitch detected in signal <%s> at time = %g\n",
		      s->name, Time);
	      s->nFH = max(s->nFH-1, 0);
	      s->state = HH;
	    }
	  break;
	  
	case XL:
	  if (val < L)
	    {
	      s->FL[s->nFL++] = linear_interp(OldTime, Time, oval, val, L);
	      s->state        = LL;
	    }
	  else if (val >= X)
	    {
	      fprintf(stderr,
		 "Warning:  glitch detected in signal <%s> at time = %g\n",
		      s->name, Time);
	      s->nFX = max(s->nFX-1, 0);
	      s->state = HX;
	    }
	  break;

	case LL:
	  if (val >= L)
	    {
	      s->RL[s->nRL++] = linear_interp(OldTime, Time, oval, val, L);
	      s->state        = LX;
	    }
	  break;

	case LX:
	  if (val >= X)
	    {
	      s->RX[s->nRX++] = linear_interp(OldTime, Time, oval, val, X);
	      s->state        = XH;
	    }
	  else if (val < L)
	    {
	      fprintf(stderr,
		 "Warning:  glitch detected in signal <%s> at time = %g\n",
		      s->name, Time);
	      s->nRL = max(s->nRL-1, 0);
	      s->state = LL;
	    }
	  break;

	case XH:
	  if (val >= H)
	    {
	      s->RH[s->nRH++] = linear_interp(OldTime, Time, oval, val, H);
	      s->state        = HH;
	    }
	  else if (val < X)
	    {
	      fprintf(stderr,
		 "Warning:  glitch detected in signal <%s> at time = %g\n",
		      s->name, Time);
	      s->nRX = max(s->nRX-1, 0);
	      s->state = LX;
	    }
	  break;

	default:
	  fprintf(stderr,"Bad switch %d in finite_state\n",s->state);
	  break;
	}
    }
  while (oldState != s->state);
}


/************************************************************************
Examine all signals being traced for possible transitions during this time
tick.  For signals that do change, linearly interpolate the time and record
it in the transition arrays.  Afterward, copy current voltage V into OldV
for use during next tick.

On entry:

************************************************************************/
void tick()
{
  int i;

  for (i = 0; i < NumTraces; i++)
    {
      finite_state(TraceList[i]);
      TraceList[i]->OldV = TraceList[i]->V;
    }
}


/************************************************************************
Print out all transitions accumulated for trace variables.
This procedure is used in debugging.
************************************************************************/
void print_transitions()
{
  static char outbuff[96];
  int done;
  int i, j;

  for (i = 0; i < NumTraces; i++)
    {
      printf("\n");
      printf("%-12s       ",TraceList[i]->name);
      printf("RL         RX         RH         FL         FX         FH\n");

      for (j = 0; j < MAX_CYCLES; j++)
	{
	  done = TRUE;

	  sprintf(&outbuff[0],"[%5d]        ",j);

	  if (j < TraceList[i]->nRL)
	    {
	      sprintf(&outbuff[15],"%8.4e   ",TraceList[i]->RL[j]);
	      done = FALSE;
	    }
	  else
	    sprintf(&outbuff[15],"           ");

	  if (j < TraceList[i]->nRX)
	    {
	      sprintf(&outbuff[26],"%8.4e   ",TraceList[i]->RX[j]);
	      done = FALSE;
	    }
	  else
	    sprintf(&outbuff[26],"           ");

	  if (j < TraceList[i]->nRH)
	    {
	      sprintf(&outbuff[37],"%8.4e   ",TraceList[i]->RH[j]);
	      done = FALSE;
	    }
	  else
	    sprintf(&outbuff[37],"           ");

	  if (j < TraceList[i]->nFL)
	    {
	      sprintf(&outbuff[48],"%8.4e   ",TraceList[i]->FL[j]);
	      done = FALSE;
	    }
	  else
	    sprintf(&outbuff[48],"           ");

	  if (j < TraceList[i]->nFX)
	    {
	      sprintf(&outbuff[59],"%8.4e   ",TraceList[i]->FX[j]);
	      done = FALSE;
	    }
	  else
	    sprintf(&outbuff[59],"           ");

	  if (j < TraceList[i]->nFH)
	    {
	      sprintf(&outbuff[70],"%8.4e \n",TraceList[i]->FH[j]);
	      done = FALSE;
	    }
	  else
	    sprintf(&outbuff[70],"           \n");
	  
	  if (done)
	    break;
	  else
	    printf("%s",outbuff);
	}
    }
}
