/* tree.c --- parse tree manipulation routines for tcalc

MODULE
    tree.c

DESCRIPTION
    This module contains utility routines for manipulating and evaluating
    the parse tree in the tcalc interpretter.

HISTORY
    6/6/88 --- written by Steven Molnar
BUGS

---------------------------------------------------------*/

#include <strings.h>
#include "tcalc.h"
#include "parse.h"

extern SIGNAL *TraceList[];	 /* table of signals being traced */
extern int     NumTraces;

extern VAR    *VarList[];        /* variables */
extern int     NumVars;


/************************************************************************
Evaluate a transition time specified in trans.

On entry:
  trans = pointer to TRANS record describing transition
On exit:
  return floating point (time) value of transition
************************************************************************/
float transition_eval(trans)
     TRANS *trans;
{
  switch (trans->trans)
    {
    case LOW:
      if (trans->rf == RISE)
	if (trans->num < trans->signal->nRL)
	  return(trans->signal->RL[trans->num]);
	else
	  fprintf(stderr,"Transition <%s: R, L, %d> not found\n",
		  trans->signal->name, trans->num);
      else
	if (trans->num < trans->signal->nFL)
	  return(trans->signal->FL[trans->num]);
	else
	  fprintf(stderr,"Transition <%s: F, L, %d> not found\n",
		  trans->signal->name, trans->num);
      return(0.0);
      
    case CROSS:
      if (trans->rf == RISE)
	if (trans->num < trans->signal->nRX)
	  return(trans->signal->RX[trans->num]);
	else
	  fprintf(stderr,"Transition <%s: R, X, %d> not found\n",
		  trans->signal->name, trans->num);
      else
	if (trans->num < trans->signal->nFX)
	  return(trans->signal->FX[trans->num]);
	else
	  fprintf(stderr,"Transition <%s: F, X, %d> not found\n",
		  trans->signal->name, trans->num);
      return(0.0);
      
    case HIGH:
      if (trans->rf == RISE)
	if (trans->num < trans->signal->nRH)
	  return(trans->signal->RH[trans->num]);
	else
	  fprintf(stderr,"Transition <%s: R, H, %d> not found\n",
		  trans->signal->name, trans->num);
      else
	if (trans->num < trans->signal->nFH)
	  return(trans->signal->FH[trans->num]);
	else
	  fprintf(stderr,"Transition <%s: F, H, %d> not found\n",
		  trans->signal->name, trans->num);
      return(0.0);
      
    default:
      fprintf(stderr,"Bad switch %d in transition_eval\n",
	      trans->trans);
      return(0.0);
    }
}


/************************************************************************
Evaluate the expression contained in expr.  Return the resulting value.

On entry:
  expr = expression to evaluate
On exit:
  return floating point value of expression
************************************************************************/
float expr_eval(expr)
     EXPR *expr;
{
  float temp;

  switch (expr->type)
    {
    case MIN:
      return(min(expr_eval(expr->arg1), expr_eval(expr->arg2)));
    case MAX:
      return(max(expr_eval(expr->arg1), expr_eval(expr->arg2)));
    case PLUS:
      return(expr_eval(expr->arg1) + expr_eval(expr->arg2));
    case MINUS:
      return(expr_eval(expr->arg1) - expr_eval(expr->arg2));
    case MULT:
      return(expr_eval(expr->arg1) * expr_eval(expr->arg2));
    case DIVIDE:
      temp = expr_eval(expr->arg2);
      if (temp != 0)
	return(expr_eval(expr->arg1) / temp);
      else
	{
	  fprintf(stderr,"Error:  division by zero\n");
	  return(0.0);
	}
    case UMINUS:
      return(-expr_eval(expr->arg1));
    case IDENT:
      if (expr->contents.var->valid)
	return(expr->contents.var->value);
      else 
	{
	  fprintf(stderr,"Error:  variable %s used before set\n",
		  expr->contents.var->name);
	  return(0.0);
	}
    case CROSS:
      return(transition_eval(&(expr->contents.trans)));
    case FLOAT:
      return(expr->contents.value);
    default:
      fprintf(stderr,"Illegal expression type: %d in expr_eval\n",expr->type);
      return(0.0);
    }
}
