/*
 * incat - concatenate .in vector files, reformatting 
 * 	subsequent files to match the format of the first one.
 *
 *	Copyright 2002 Steve Tell
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include "infile.h"
#include "regexp.h"


struct incat_map_entry {
	enum sigtype type;
	int index;
	char val;
};
void build_mapping(struct infile *mi, struct infile *inf, GArray *map);
char *remap_vectors(struct infile *mi, struct infile *inf, GArray *map, char *lastvec, int *cycle);
void transcribe_header(struct infile *inf);

static char *progname = "incat";
static int g_verbose;

static void 
usage()
{
	fprintf(stderr, "Usage:  %s [options] <infile>...\n", progname);
	fprintf(stderr, "  - concatenate and/or reformat .in vector files\n");
	fprintf(stderr, "options:\n");
	fprintf(stderr, "  -h     headeronly - output only header from first file\n");
	fprintf(stderr, "  -n     noheader - do not output any header\n");
	fprintf(stderr, "  -v     verbose debugging mode\n");

	exit(3);
}

int
main(int argc, char **argv)
{
	int i, j;
	int c;
	int headeronly = 0;
	int noheader = 0;
	extern char *optarg;
	extern int optind;
	extern int optopt;
	struct infile *master_inf;
	struct infile *inf;
	GArray *vec_map;  /* array of struct incat_map_entry */

	struct inf_signal *sig;
	struct inf_field *field;
	char *lastvec;
	gpointer gp;
	int cycle_count;

	while((c = getopt(argc, argv, "hnv")) != -1) {
		switch(c) {
		case 'n':
			noheader = 1;
			break;
		case 'h':
			headeronly = 1;
			break;
		case 'v':
			g_verbose = 1;
			break;
		default:
			fprintf(stderr, "%s: Unrecognized option: -%c\n", progname, optopt);
			usage();
			break;
		}
	}
	if(optind >= argc) {
		fprintf(stderr, "%s: no input files specified.\n", progname);
		usage();
	}

	master_inf = infile_new(argv[optind]);
	if(!master_inf) {
		perror(argv[optind]);
		exit(1);
	}
	if(!infile_parse_header(master_inf)) {
		fprintf(stderr, "%s: bad header\n", argv[optind]);
		exit(1);
	}
	if(!noheader) {
		transcribe_header(master_inf);
	}
	vec_map = g_array_new(0, 0, sizeof(struct incat_map_entry));
	cycle_count = 0;
	if(!headeronly) {
		build_mapping(master_inf, master_inf, vec_map);
		lastvec = remap_vectors(master_inf, master_inf, vec_map, 
					NULL, &cycle_count);
	}
	optind++;

	for(; optind < argc; optind++) {
		inf = infile_new(argv[optind]);
		if(!inf) {
			perror(argv[optind]);
			exit(1);
		}
		if(!infile_parse_header(inf)) {
			fprintf(stderr, "%s: bad header\n", argv[optind]);
			exit(1);
		}
		
		build_mapping(master_inf, inf, vec_map);
		remap_vectors(master_inf, inf, vec_map, lastvec, &cycle_count);
		infile_close(inf);
	}
	exit(0);
}

/*
 * Build the mapping needed for fast remapping of vector lines.
 * The map array contains one entry for each character to be generated in
 * each output vector line.  Map entries can be 
 *	a constant bit, indicated by type=STATSIG and the value
 * 	a space, indicated by type=STATSIG and value ' '
 *	a vector element, indicated by type=INSIG or type=OUTSIG
 *		for these, the index indicates which bit of the 
 *		source file's input or output vector to use.
 */
void build_mapping(struct infile *mi, struct infile *inf, GArray *map)
{
	struct incat_map_entry ent;
	struct inf_field *field;
	struct inf_signal *msig, *sig;
	int i, j;
	gpointer gp;
	/* first, clear the old mapping, if any */
	g_array_set_size(map, 0);

	/* inputs */
	for (i = 0; i < mi->inputs.fields->len; i++) {
		field =  (struct inf_field *)
			g_ptr_array_index(mi->inputs.fields, i);

		for (j = 0; j < field->nsigs; j++) {
			msig =  (struct inf_signal *)
				g_ptr_array_index(mi->inputs.sigs, field->firstsig+j);
			if(gp = g_hash_table_lookup(inf->statics.sighash,
						    msig->name)) {
				sig =  (struct inf_signal *)gp;
				ent.type = STATSIG;
				ent.index = 0;
				ent.val = sig->val ? '1' : '0';
			} else if(gp = g_hash_table_lookup(inf->inputs.sighash,
						    msig->name)) {
				sig =  (struct inf_signal *)gp;
				ent.type = INSIG;
				ent.index = sig->index;
				ent.val = 'x';
			} else {
				ent.type = UNKNOWN;
				ent.index = 0;
				ent.val = 'x';
			}
			g_array_append_val(map, ent);
		}
		/* space seperator */
		ent.type = STATSIG;
		ent.index = 0;
		ent.val = ' ';
		g_array_append_val(map, ent);
	}
	ent.type = STATSIG;
	ent.index = 0;
	ent.val = ' ';
	g_array_append_val(map, ent);
	
	/* outputs */
	for (i = 0; i < mi->outputs.fields->len; i++) {
		field =  (struct inf_field *)
			g_ptr_array_index(mi->outputs.fields, i);

		for (j = 0; j < field->nsigs; j++) {
			msig =  (struct inf_signal *)
				g_ptr_array_index(mi->outputs.sigs, field->firstsig+j);
			if(gp = g_hash_table_lookup(inf->outputs.sighash,
						    msig->name)) {
				sig =  (struct inf_signal *)gp;
				ent.type = OUTSIG;
				ent.index = sig->index;
				ent.val = 'x';
			} else {
				ent.type = UNKNOWN;
				ent.index = 0;
				ent.val = 'x';
			}
			g_array_append_val(map, ent);
		}
		/* space seperator */
		ent.type = STATSIG;
		ent.index = 0;
		ent.val = ' ';
		g_array_append_val(map, ent);
	}
	
	if(g_verbose) {
		struct incat_map_entry *entp;
		printf("# maptable %s -> %s:\n",
		       inf->filename, mi->filename);
		for(i = 0; i < map->len; i++) {
			entp = &g_array_index(map, struct incat_map_entry, i);
			switch(entp->type) {
			case STATSIG:
				printf("# [%d] <- '%c'\n", i, entp->val);
				break;
			case INSIG:
			case OUTSIG:
				printf("# [%d] <- %d[%i]\n", i, entp->type, entp->index);
				break;
			case UNKNOWN:
				printf("# [%d] unknown\n", i);
				break;
			default:
				printf("# [%d] bad type %d\n", entp->type);
				break;
			}
		}
	}
}

/*
 * map vectors from source file to master file, using the map.
 * Returns the last vector processed, as potential starting point for
 * mapping further files.
 */
char *
remap_vectors(struct infile *mi, struct infile *inf, GArray *map, 
	      char *lastvec, int *cycle)
{
	struct incat_map_entry *ent;
	char *invec, *outvec;
	int inflag, outflag;
	int edge;
	char *comment;
	int status;

	int i, j;

	if(!lastvec) {
		lastvec = g_new0(char, map->len+1);
		memset(lastvec, 'x', map->len);
		lastvec[map->len] = 0;
	}

	invec = g_new(char, inf->inputs.sigs->len);
	outvec = g_new(char, inf->outputs.sigs->len);

	while((status = infile_parse_vector(inf, &inflag, invec, &outflag, 
					    outvec, &edge, &comment) == 0)) {
		if(inflag) {
			*cycle += 1;
			for(i = 0; i < map->len; i++) {
				ent = &g_array_index(map, struct incat_map_entry, i);
				switch(ent->type) {
				case STATSIG:
					lastvec[i] = ent->val;
					break;
				case INSIG:
					lastvec[i] = invec[ent->index];
					break;
				case OUTSIG:
					if(outflag) {
						lastvec[i] = outvec[ent->index];
					} else {
						lastvec[i] = 'x';
					}
					break;
				}
			}
			printf("%s %c", lastvec, edge == EDGE_RISE ? 'r' : 'f');
		} 
		if(comment && *comment) {
			if(inflag) {
				printf(" %d # %s", *cycle, comment);
			} else {
				printf("#%d  %s", *cycle, comment);
			}
		} else if(inflag) {
			printf(" %d", *cycle);
		}
		putchar('\n');
	}

	if(status < 0)
		infile_error(inf, 0, "syntax error in vectors");

	g_free(invec);
	g_free(outvec);
	return lastvec;
}

/*
 * copy a .in file's header to output.
 * in order to preserve comments, we rewind the file and copy the text,
 * instead of regenerating the header.
 */
void transcribe_header(struct infile *inf)
{
	regexp *vechead;
	int done = 0;
	char *line;
	char *headname;

	vechead = regcomp("^(vectors):[ \t\n]*$");  /* will exit if error */
	assert(vechead);

	fseek(inf->fp, 0, SEEK_SET);
	inf->lineno = 0;

	while(!done) {
		/* Read next line, looking for section keyword */
		if(fread_line(inf->fp, &inf->linebuf, &inf->lbufsize) == EOF) {
			infile_error(inf, 1, "unexpected EOF eof in header");
			return;
		}
		inf->lineno++;
		line = inf->linebuf;

		if(regexec(vechead, line)) {
			done = 1;
		}
		printf("%s\n", line);
	}
	if(g_verbose) {
		printf("# transcribe_header done\n");
      	}
}
