/*
 *
 * infile.c -- parser routines for doirsim/dospice ".in" vector file
 * by Steve Tell
 *
 */

#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <ctype.h>
#include <assert.h>
#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif
#include <sys/stat.h>
#include <sys/types.h>
#include "infile.h"
#include "regexp.h"

/* Stuff for name=value attributes on signal fields */
static void parse_attribute_ignore(struct infile *inf, int id, char *value, struct inf_signal *sigt, struct inf_field *fld);
static void parse_attribute_etfo(struct infile *inf, int id, char *value, struct inf_signal *sigt, struct inf_field *fld);
static void parse_attribute_format(struct infile *inf, int id, char *value, struct inf_signal *sigt, struct inf_field *fld);
static void parse_attribute_probe(struct infile *inf, int id, char *value, struct inf_signal *sigt, struct inf_field *fld);

struct sfattr {
	int a_id;
	char *a_name;
	ATTRCB a_func;
};
static struct sfattr AttrTab[] = {
	{SATTR_ETFO,	"ETFO",   parse_attribute_etfo},
	{SATTR_IGNORE,	"IGNORE", parse_attribute_ignore},
	{SATTR_FORMAT,	"FORMAT", parse_attribute_format}, 
	{SATTR_VPORT,	"VPORT",  NULL},
	{SATTR_PROBE,	"PROBE",  parse_attribute_probe},
	{0, NULL, NULL},
} ;

/* stuff for extensible handling of header sections */
static void handle_sect_static(struct infile *inf, int id, char *line);
static void handle_sect_iom(struct infile *inf, int id, char *line);
static void handle_sect_clocks(struct infile *inf, int id, char *line);

struct sectht {
	int id;
	char *name;
	INSCB func;
};
static struct sectht SectTab[] = {
	{INSECT_UNKNOWN, "no-section",	NULL},
	{INSECT_STATIC,	"static",	handle_sect_static}, 
	{INSECT_INPUTS,	"inputs", 	handle_sect_iom},
	{INSECT_OUTPUTS,"outputs", 	handle_sect_iom},
	{INSECT_MONITOR,"monitor", 	handle_sect_iom},
	{INSECT_VECTORS,"vectors", 	NULL},
	{INSECT_CLOCKS,	"clocks", 	handle_sect_clocks}, 
	{INSECT_SPICE,	"spice", 	NULL}, 
	{INSECT_IRSIM,	"irsim", 	NULL},  
	{INSECT_VERILOG,"verilog", 	NULL},
	{0, NULL, NULL},
} ;

/* Global routines *********************************************************/

/*
 * set parsing routine for an attribute
 * If an attribute is only used by one client of the inparse library,
 * the parsing routine is move there and the pointers set up here.
 */
void infile_set_attribute_parser(int id, ATTRCB afunc)
{
	int i;
	for(i = 0; AttrTab[i].a_name; i++) {
		if(id == AttrTab[i].a_id)
			AttrTab[i].a_func = afunc;
	}
}

/*
 * set handler for a custom header section.
 */
void infile_set_section_handler(int id, INSCB afunc)
{
	int i;
	for(i = 0; SectTab[i].name; i++) {
		if(id == SectTab[i].id)
			SectTab[i].func = afunc;
	}
}

/*
 * utility function to read whole line into buffer, expanding buffer if needed.
 * line buffer must be allocated with g_malloc/g_new, or NULL in which case
 * we allocate an initial, buffer.
 * returns 0 or EOF.
 */
int
fread_line(FILE *fp, char **bufp, int *bufsize)
{
	int c;
	int n = 0;
	if(*bufp == NULL) {
		if(*bufsize == 0)
			*bufsize = 1024;
		*bufp = g_new(char, *bufsize);
	}
	while(((c = getc(fp)) != EOF) && c != '\n') {
		(*bufp)[n++] = c;
		if(n >= *bufsize) {
			*bufsize *= 2;
			*bufp = g_realloc(*bufp, *bufsize);
		}
		if(c == '\n')
			break;
	}
	(*bufp)[n] = 0;
	if(c == EOF)
		return EOF;
	else
		return 0;
}

/* object-method routines **************************************************/

/*
 * open an infile and allocate the object for it, but don't read anything.
 * returns pointer to struct infile or NULL on error.
 */
struct infile *infile_new(char *filename)
{
	struct infile *inf;
	FILE *fp;
	fp = fopen(filename, "r");
	if(!fp)
		return NULL;
	inf = g_new0(struct infile, 1);
	inf->filename = strdup(filename);
	inf->fp = fp;

	inf->statics.sigs = g_ptr_array_new();
	inf->statics.fields = g_ptr_array_new();
	inf->inputs.sigs = g_ptr_array_new();
	inf->inputs.fields = g_ptr_array_new();
	inf->outputs.sigs = g_ptr_array_new();
	inf->outputs.fields = g_ptr_array_new();
	inf->monitors.sigs = g_ptr_array_new();
	inf->monitors.fields = g_ptr_array_new();
	
	/* hash table of signals; key is signal name (string)
	 * value is struct inf_signal
	 */
	inf->statics.sighash = g_hash_table_new(g_str_hash, g_str_equal);
	inf->inputs.sighash = g_hash_table_new(g_str_hash, g_str_equal);
	inf->outputs.sighash = g_hash_table_new(g_str_hash, g_str_equal);
	inf->monitors.sighash = g_hash_table_new(g_str_hash, g_str_equal);
	return inf;
}

void
infile_close(struct infile *inf)
{
	fclose(inf->fp);
	g_free(inf->filename);
	g_ptr_array_free(inf->statics.sigs, 1);
	g_ptr_array_free(inf->statics.fields, 1);
	g_ptr_array_free(inf->inputs.sigs, 1);
	g_ptr_array_free(inf->inputs.fields, 1);
	g_ptr_array_free(inf->outputs.sigs, 1);
	g_ptr_array_free(inf->outputs.fields, 1);
	g_ptr_array_free(inf->monitors.sigs, 1);
	g_ptr_array_free(inf->monitors.fields, 1);

	g_hash_table_destroy(inf->statics.sighash);
        g_hash_table_destroy(inf->inputs.sighash);
	g_hash_table_destroy(inf->outputs.sighash);
	g_hash_table_destroy(inf->monitors.sighash);

	g_free(inf);
}

/*
 * Print a formatted error message and maybe die.
 *
 * inputs:
 *   die = 0 to return, 1 to exit(2)
 *   fmt = printf-style format string 
 *   ... arguments to print
*/
void infile_error(struct infile *inf, int die, char *fmt, ...)
{
	va_list ap;
	fprintf(stderr, "\"%s\",%d: ", inf->filename, inf->lineno);
	va_start(ap, fmt);
	vfprintf(stderr, fmt, ap);
	va_end(ap);
	putc('\n', stderr);
	if (die)
		exit(2);
}

/*
 * estimate number of vectors of data in a .in file, based on
 * the number of bits required to be on each line and the file size
 *
 * Must be called after parse_infile_header()
 */
int infile_num_vectors(struct infile *inf, int outspresent)
{
	int llen;	// minimum line length for vector lines
	int nvecs;	// maximum number of vectors in file
	struct stat st;

	llen = inf->inputs.sigs->len + inf->inputs.fields->len;
	if(outspresent) 
		llen += inf->outputs.sigs->len + inf->outputs.fields->len;
	
	if(fstat(fileno(inf->fp), &st) < 0) {
		fprintf(stderr, "fstat(%s): %s\n", inf->filename, strerror(errno));
		exit(2);
	}
	nvecs =  st.st_size / llen + 1;
	return nvecs;
}

/*
 * Recursive routine to expand names possibly containing iterators.
 * On first call (whith name != NULL), initialize variables and 
 * return first name.  On subsequent calls, return next names.
 * 
 * On entry:
 *   name  = (first call) signal name possibly containing interators 
 *           (subsequent calls) NULL
 *   level = level at which this function has been called
 * On exit:
 *   oname =  flattened signal name
 *   rtn value = 0 if more names to go, 1 if last name
*/

static int next_sig_name(struct infile *inf, char *iname, 
			 char *oname, int level)
{
  // Stack for recursion
  static char prefix [MAX_ITERATORS][1024]; // name up until iterator
  static int  first  [MAX_ITERATORS];		    // iterator start val
  static int  last   [MAX_ITERATORS];		    // iterator end val
  static int  curr   [MAX_ITERATORS];		    // iterator cur val
  static int  digits [MAX_ITERATORS];		    // min. # of digits in iterator
  static int  more   [MAX_ITERATORS];		    // further levels?

  char *ptr, *firstptr, *lastptr;
  int   bump;

  // Check for overflow
  if (level >= MAX_ITERATORS)
    infile_error(inf, 1, "Maximum number of iterators exceeded");

  //
  // If first call load variables
  // 
  if (iname != NULL)
    {
      // Make local copy of name
      strcpy(prefix[level], iname);
      
      // Check for iterator
      if ((ptr = strchr(prefix[level], '{')) != NULL)
	{
	  // Prefix ends at iterator
	  *ptr = '\0';

	  // Parse the first number
	  firstptr = ptr+1;
	  if ((ptr = strchr(firstptr, ':')) == NULL)
	    infile_error(inf, 1, "Syntax error in iterator");
	  *ptr = '\0';
	  digits[level] = strlen(firstptr);
	  first [level] = atoi(firstptr);

	  // Parse the last number
	  lastptr = ptr+1;
	  if ((ptr = strchr(lastptr, '}')) == NULL)
	    infile_error(inf, 1, "Syntax error in iterator");
	  *ptr = '\0';
	  if ((int)strlen(lastptr) < digits[level])
	    digits[level] = strlen(lastptr);
	  last[level] = atoi(lastptr);

	  // Check if further calls needed
	  if (strlen(++ptr) > 0)
	    more[level] = 1;
	  else 
	    more[level] = 0;
	}
      else
	{
	  first [level]  = -1;
	  last  [level]  = -1;
	  more  [level]  = 0;
	}
      
      curr [level] = first[level];
    }

  // Add this level's contributation to oname
  strcat(oname, prefix[level]);
  if (curr[level] >= 0)
    {
	    char iter[1024];
	    int  i;

      sprintf(iter, "%d", curr[level]);
      for (i = strlen(iter); i < digits[level]; i++)
	strcat(oname, "0");
      strcat(oname, iter);
    }

  // Get contribution from lower levels if necessary
  if (more[level])
    bump = next_sig_name(inf, (iname == NULL) ? NULL: ptr, oname, level+1);
  else
    bump = 1;    // always bump if this is the bottom level
  
  // Update iterator if necessary
  if (bump)
    {
      // Check if done
      if (curr[level] == last[level])
	{
	  curr[level] = first[level];
	  return(1);
	}

      if (first[level] < last[level])
	curr[level] += 1;
      else 
	curr[level] -= 1;
      return(0);
    }
  
  // More iterations to go
  return(0);

}

/*
 * next_name_alloc - get next name from iterator, allocating storage for
 *	the name.
 *	calls next_sig_name with a static buffer and then copies.
 *
*/
static int next_name_alloc(struct infile *inf, char *iname, char **oname, int level)
{
	char buf[1024];
	int done;
	buf[0] = 0;

	done = next_sig_name(inf, iname, buf, level);
	*oname = strdup(buf);
	if(!*oname)
		infile_error(inf, 1, "strdup: out of memory in next_name_alloc");
	return done;
}



/*
 * parse IGNORE attribute on signal field
 */
static void parse_attribute_ignore(struct infile *inf, int id, char *value, 
				   struct inf_signal *sigt,
				   struct inf_field *fld)
{
	if(strcasecmp(value, "NONE") == 0)
		sigt->ignore = IGNORE_NONE;
	else if (strcasecmp(value, "DELAY") == 0)
		sigt->ignore = IGNORE_DELAY;
	else if (strcasecmp(value, "SPICE") == 0)
		sigt->ignore = IGNORE_SPICE;
	else if (strcasecmp(value, "CAZM") == 0)
		sigt->ignore = IGNORE_SPICE;
	else if (strcasecmp(value, "ALL") == 0)
		sigt->ignore = IGNORE_ALL;
	else if (strcasecmp(value, "IRSIM") == 0)
		sigt->ignore = IGNORE_IRSIM;
	else
		infile_error(inf, 1, "garbled IGNORE attribute: %s", value);
}
/*
 * parse ETFO attribute on signal field
 */
static void parse_attribute_etfo(struct infile *inf, int id, char *value,
				 struct inf_signal *sigt,
				 struct inf_field *fld)
{
	char *token;

	token = strtok(value, "+\n");
	if(!token) {
		infile_error(inf, 1, "syntax error in ETFO: fanout expected");
	}
	sigt->fanout = atof(token);
	token = strtok(NULL, ",\n");
	if(token)
		sigt->delay = atof(token);
	token = strtok(NULL, ",\n");
	if(token) {
		switch(toupper(*token)) {
		case 'F':
			sigt->edgestyle = EDGE_FALL;
			break;
		case 'R':
			sigt->edgestyle = EDGE_RISE;
			break;
		case 'B':
			sigt->edgestyle = EDGE_BOTH;
			break;
		default:
			infile_error(inf, 1, "edge style must be F, R, or B");
			break;
		}
	}
	token = strtok(NULL, " \t\n");
	if(token) {
		sigt->wavename = strdup(token);
	}
}
/*
 * parse FORMAT attribute on signal field
 */
static void parse_attribute_format(struct infile *inf, int id, char *value,
				   struct inf_signal *sigt,
				   struct inf_field *fld)

{
	fld->leadzero = 0;  /* default except for %b */
	for( ; *value; value++) {
		switch(*value) {
		case '%':
			break;
		case '0':
			fld->leadzero = 1;
			break;
		case 'b':
		case 'd':
		case 'u':
		case 'o':
			fld->conv = *value;
			break;
		case 'h':
		case 'x':
			fld->conv = 'h';
			break;
		default:
			infile_error(inf, 1, "illegal format specfication");
			break;
		}
	}
}
/*
 * parse PROBE attribute on signal field
 */
static void parse_attribute_probe(struct infile *inf, int id, char *value, 
				   struct inf_signal *sigt,
				   struct inf_field *fld)
{
	fld->probestr = strdup(value);
}
/*
 * Parse signal attributes for either inputs or outputs section
 * 
 * On entry:
 *   line = output line
 * On exit:
 *	Field pointed to by fld filled in with all required info
 *		for the set of signals on this line
 *	template Sig pointed to by sigt filled in with common information,
 *		 ready to be copied to Sig's of actual signals.
 *
 * Side Effects:
 *	line will be terminated with '\0' before attributes; only signals
 *	will remain.
 *	inf->ActiveEdge updated.
*/
static void parse_attributes(struct infile *inf, char *line, 
				   struct inf_signal *sigt,
				   struct inf_field *fld)
{
	int i;
	char *token;
	char *cp;
	char *attrs;
	int found_attr;
	sigt->ignore = IGNORE_NONE;
	sigt->fanout = DEFAULT_FANOUT;
	sigt->delay = DEFAULT_DELAY;
	sigt->edgestyle = EDGE_FALL;
	sigt->wavename = NULL;
	sigt->name = NULL;

	fld->leadzero = 1;
	fld->conv = 'b';
  
	// find attributes section, if any, by looking for '=' 
	// in first name=value attribute.
	cp = strchr(line, '=');
	if(cp) {
		// back up to beginning of first name=value pair by looking for
		// whitespace
		for(attrs = cp; !isspace(*attrs) && attrs > line; attrs--)
			;
		if(attrs <= line) {
			infile_error(inf, 1, "syntax error: signal name expected");
		}
		*attrs++ = 0;

		while(attrs) {
			// all this crud because strtok isn't reentrant
			char *pname = attrs;
			char *pval;
			pval = strchr(pname, '=');

			if(!pval) {
				infile_error(inf, 1, "syntax error: parameter value expected");
			}
			*pval++ = 0;	// terminates pname
			attrs = strpbrk(pval, " \t\n");
			if(attrs) {
				*attrs++ = 0;
				while(*attrs && isspace(*attrs))
					attrs++;
				if(*attrs == '\0')
					attrs = NULL;
			}
			
			// pname is attribute name,
			// pval is attribute value
			found_attr = 0;
			for(i = 0; AttrTab[i].a_name; i++) {
				if(strcasecmp(pname, AttrTab[i].a_name) == 0) {
					found_attr = 1;
					if(AttrTab[i].a_func)
						(AttrTab[i].a_func)(inf, i, pval, sigt, fld);
				}
			}
			if(!found_attr) {
				infile_error(inf, 1, "unknown attribute %s=%s", pname, pval);			
			}
		} /* end while(attrs) */
		inf->ActiveEdge |= sigt->edgestyle;
	} /* end if(attrs section) */
}

/*
 * post-parsing setup required for a Field structure
 * - computes the field's column width based on number of bits and
 *   format.
 * 
*/


static void field_ppsetup(struct infile *inf, struct inf_field *fld)
{

	// number of number of digits to display n-bit decimal number, 0 <= n <= 64
	static short ltab[] = {
		0,1,1,1,2,2,2,3,3,3,4,4,4,4,5,5,5,6,6,6,7,7,7,7,
		8,8,8,9,9,9,10,10,10,10,11,11,11,12,12,12,
		13,13,13,13,14,14,14,15,15,15,16,16,16,16,
		17,17,17,18,18,18,19,19,19,19,20 };

	switch(fld->conv) {
		case 'b':
			fld->width = fld->nsigs;
			break;
		case 'd':
		case 'u':
			if(fld->nsigs > 32) {
				infile_error(inf, 1, "Decimal output format not implemented for busses wider than 32 bits\n");
			}
			fld->width = ltab[fld->nsigs];
			break;
		case 'o':
			fld->width = (fld->nsigs-1)/3+1;
			break;
		case 'h':
			fld->width = (fld->nsigs-1)/4+1;
			break;
		default:
			infile_error(inf, 1, "Internal error fld->conv=%c\n", fld->conv);
			break;
	}
}

/* 
 * parse a line in the clocks section.
 */
static void handle_sect_clocks(struct infile *inf, int id, char *line)
{
}
/*
 * Parse a line in the static inputs section
 * 
 * On entry:
 *   line = input line
*/
static void handle_sect_static(struct infile *inf, int id, char *line)
{
	char *token;
	int nsigs;
	int firstsig;
	int i;

	/* Remove comments */
	token = strchr(line, '#');
	if (token != NULL)
		*token = '\0';
  
	/* Check for blank line */
	token = strtok(line, " \t\n");
	if (token == NULL)
		return;
  
	/* Process each signal on the line */
	nsigs = 0;
	firstsig = inf->statics.sigs->len;
	do {
		int firstcall;
		int done;
		char *signame;
		struct inf_signal *sig;
      
		/* Store instances of signal name */
		firstcall = 1;
		do {
			done = next_name_alloc(inf, (firstcall==1) ? token : NULL, 
					       &signame, 0);
			sig = g_new0(struct inf_signal, 1);
			sig->name = signame;
			sig->type = STATSIG;
			g_ptr_array_add(inf->statics.sigs, sig);
			g_hash_table_insert(inf->statics.sighash, 
					    sig->name, sig);
			nsigs++;
			firstcall = 0;
		} while (!done);
      
		/* Get next signal name or iterator */
		if ((token = strtok(NULL, " \t\n")) == NULL)
			infile_error(inf, 1, "bit vector not found for static signals");
      
	} while (strspn(token, "01") < strlen(token));
  
	// Make sure bit vector has same length as number of signals on line
	if (nsigs != (int)strlen(token))
		infile_error(inf, 1, "wrong vector length for static signal");
  
	// Record bit values with signal names
	for (i = 0; i < (int)strlen(token); i++)
	{
		struct inf_signal *sig;
		sig = (struct inf_signal *)
			g_ptr_array_index(inf->statics.sigs, firstsig+i);

		if (token[i] == '0')
			sig->val = 0;
		else if (token[i] == '1')
			sig->val = 1;
		else
			infile_error(inf, 1, "static signal format error");
	}
}


/*
 * Parse a line in a signal spec section (input, output, monitor, etc).
 * 
 * On entry:
 *   char *line = line read from .in file
 *   char *ssname = name of signal set being read, for use in messages
 *   SigSet *ss = pointer to structure containing Sigs and Fields arrays 
*/
static void handle_sect_iom(struct infile *inf, int sect, char *line) 
{
	char *token;
	struct inf_sigset *ss;
	struct inf_signal sigtmp;
	struct inf_signal *sig;
	struct inf_field *field;
	enum sigtype stype;
  
	/* figure which sigset we're in */
	switch(sect) {
	case INSECT_INPUTS:
		ss = &inf->inputs;
		stype = INSIG;
		break;
	case INSECT_OUTPUTS:
		ss = &inf->outputs;
		stype = OUTSIG;
		break;
	case INSECT_MONITOR:
		ss = &inf->monitors;
		stype = MONSIG;
		break;
	default:
		infile_error(inf, 1, "internal: bad section at %s:%d", __FILE__, __LINE__);
		break;
	}

	/*/ Remove comments */
	token = strchr(line, '#');
	if (token != NULL)
		*token = '\0';

	field = g_new0(struct inf_field, 1);
	parse_attributes(inf, line, &sigtmp, field);

	token = strtok(line, " \t\n");
	if (token == NULL)
		return;
  
	// Process each signal name field
	field->name = strdup(token);
	field->nsigs = 0;
	field->firstsig = ss->sigs->len;

	do { 	/* process each item on the line */
		int firstcall;
		int done;
		char *signame;

		firstcall = 1;
		do {	/* process expanded names from each item */
			sig = g_new0(struct inf_signal, 1);
			*sig = sigtmp;    /* copy attributes */

			done = next_name_alloc(inf, (firstcall == 1) ? token : NULL, 
					       &signame, 0);
			sig->type = stype;
			sig->name = signame;
			sig->index = ss->sigs->len;
			/* add new signal structure to arrays */
			g_ptr_array_add(ss->sigs, sig);
			g_hash_table_insert(ss->sighash, sig->name, sig);
			field->nsigs++;
			firstcall = 0;
		} while (!done);
      
	} while ((token = strtok(NULL, " \t\n")) != NULL);

	field_ppsetup(inf, field);
	
	/* add our new field to the array */
	g_ptr_array_add(ss->fields, field);
}


/*
 * Routine to parse infile header
 *
 * On entry:
 *  infile object fresly created by infile_new()
 * On exit:
 *	Returns 0 if error encountered, 1 on success.
 *	If successful, signal and field elements filled in
 *	input file open and positioned to start parsing vectors
 *	
 *	If 0 is returned, state is indeterminate; caller should immediately
 *	call infile_close
 */
int infile_parse_header(struct infile *inf)
{
	int cursect;
	regexp *secthead;
	char *headname;
	int seensects;
	int done = 0;
	char *line;
	int i;

	secthead = regcomp("^([a-zA-Z]+):[ \t\n]*$");  /* will exit if error */
	assert(secthead);
	cursect = INSECT_UNKNOWN;
	seensects = 0;

/* Read and parse header sections
   sections start with a line containing "keyword:" */
	while(cursect != INSECT_VECTORS) {
		/* Read next line, looking for section keyword */
		if(fread_line(inf->fp, &inf->linebuf, &inf->lbufsize) == EOF) {
			infile_error(inf, 1, "unexpected EOF eof in header");
			return 0;
		}
		inf->lineno++;
		line = inf->linebuf;

		if(regexec(secthead, line) == 0) { 
			/* parse line to current section, if any */
			if(SectTab[cursect].func) {
				SectTab[cursect].func(inf, cursect, line);
			}
		} else { /* line matches section name pattern */
			cursect = 0;
			*secthead->endp[1] = 0;
			headname = secthead->startp[1];
			for(i = 0; SectTab[i].name; i++) {
				if(strcasecmp(headname, SectTab[i].name) == 0) {
					cursect = i;
					break;
				}
			}
			if(!cursect) {
				infile_error(inf, 0, "unknown section header \"%s:\"", headname);
				return 0;
			}
			seensects |= 1<<cursect;
		}
	}
	if(inf->ActiveEdge == EDGE_DEFAULT)
		inf->ActiveEdge = EDGE_FALL;

	if((seensects & ((1<<INSECT_INPUTS)|(1<<INSECT_OUTPUTS)) )
	   != ((1<<INSECT_INPUTS)|(1<<INSECT_OUTPUTS))) {
		infile_error(inf, 1, ".in file must have at least one input and one output");
		return 0;
	}
	return 1;
}

  
/*
 * Routine to parse next input vector from .in file
 * 
 * On exit:
 *   inflag  = set if input vectors found
 *   invec   = output vector array
 *   outflag = set if output vectors found
 *   outvec  = output vector array
 *   edgep:  if non-NULL, set to vector edge style, or 0 if none specified.
 *   comment: if non-NULL, set to pointer into line to start of comment,
 *		or to NULL if there is no comment.
 *
 *   Return value = 0 if okay
 *                  1 if EOF
 *                 -1 if non-fatal format error
*/
int infile_parse_vector(struct infile *inf, 
			int *inflag,  char *invec,
			int *outflag, char *outvec,
			int *edgep, char **comment)
{
	char  *ptr;
	char  *token;
	int    i;
	int nbits;
	int l;	
	int vecedge = 0;

	// Set defaults
	*inflag  = 0;
	*outflag = 0;
	
	// Read line, check for EOF
	if (fread_line(inf->fp, &inf->linebuf, &inf->lbufsize) == EOF)
		return(1);
	inf->lineno++;
	
	/* terminate at comment, and give caller pointer into the comment */ 
	if ((ptr = strchr(inf->linebuf, '#')) != NULL) {
		*ptr++ = '\0';
		if(comment) {
			while(isspace(*ptr))
				ptr++;
			if(*ptr)
				*comment = ptr;
			else
				*comment = NULL;
			ptr = strchr(ptr, '\n');
			if(ptr)
				*ptr = '\0';
		}
	} else {
		if(comment)
			*comment = NULL;
	}

	// If no input vectors, return now
	token = strtok(inf->linebuf, " \t\n");
	if (token == NULL)
		return(0);
  
	// Read input vectors
	invec[0] = '\0';
	for (i = 0; i < inf->inputs.fields->len; i++) {
		struct inf_field *field;
		field = (struct inf_field *)g_ptr_array_index(inf->inputs.fields, i);
		if (token == NULL)
			infile_error(inf, 1, "input vector has wrong number of fields");

		g_strup(token);
		// Copy field into return variable
		nbits = strspn(token, "01XZ-");
		if (nbits != field->nsigs)
			infile_error(inf, 1, "input vector field %d: found %d bits expected %d",
				     i, nbits, field->nsigs);

		strcat(invec, token);
      
		// Parse next field
		token = strtok(NULL, " \t\n");
	}  
	*inflag = 1;

	if (token && strspn(token, "01XxZz-")>0 ) { // Output vectors present
		*outflag = 1;
		outvec[0] = '\0';
		for (i = 0; i < inf->outputs.fields->len; i++) {
			struct inf_field *field;
			field = (struct inf_field *)g_ptr_array_index(inf->outputs.fields, i);
			if (token == NULL) {
				infile_error(inf, 0, "output vector has wrong number of fields");
				return(-1);
			}

			// Copy field into return variable, change 'x's to 'X's
			g_strup(token);
			l = strspn(token, "01XZ-");
			if (l != field->nsigs) {
				infile_error(inf, 0, "output vector field %d: found %d bits expected %d",
					     i, l, field->nsigs);
				return(-1);
			}

			strcat(outvec, token);
      
			// Parse next field
			token = strtok(NULL, " \t\n");
		}  
	} else {
		// no outvectors present; set outvec to all 'X's
		for (i = 0; i < inf->outputs.fields->len; i++)
			outvec[i] = 'X';
		outvec[i] = '\0';
	}

	vecedge = 0;
	if (token != NULL) {
		switch(toupper(*token)) {
		case 'F':
			vecedge = EDGE_FALL;
			break;
		case 'R':
			vecedge = EDGE_RISE;
			break;
		default:
			infile_error(inf, 1, "vector edge indicator must be F or R");
			break;
		}
	}
	if(edgep)
		*edgep = vecedge;

	return(0);
}
