///////////////////////////////////////////////////////////////////////////
//
// inparse.C -- parsing routines for doirsim/docazm .in file
//
// Written 6/22/94 by Steve Molnar
//
//////////////////////////////////////////////////////////////////////////

#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <ctype.h>
#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif
#include <sys/stat.h>
#include <sys/types.h>
#include "inparse.h"
extern "C" {
#include "regexp.h"
}

// Global variables
int      StatNumSigs;		         // Number of static signals
STATSIG  StatSigs[MAX_SIGS];		 // List of static signals

SigSet	ISigs;
SigSet	OSigs;
SigSet	MonSigs;			// monitored signals

int	InfileActiveEdge;

// Input file information
static FILE *InFile;
static char  InFileName[MAX_NAME_LEN];
static char  InFileLine[MAX_LINE_LEN];
static int   InFileLineCnt = 0;

/* Stuff for name=value attributes on signal fields */
static void parse_attribute_ignore(int id, char *value, Sig *sigt, Field *fld);
static void parse_attribute_etfo(int id, char *value, Sig *sigt, Field *fld);
static void parse_attribute_format(int id, char *value, Sig *sigt, Field *fld);
static void parse_attribute_probe(int id, char *value, Sig *sigt, Field *fld);

struct sfattr {
	int a_id;
	char *a_name;
	ATTRCB a_func;
};
static struct sfattr AttrTab[] = {
	{SATTR_ETFO,	"ETFO",   parse_attribute_etfo},
	{SATTR_IGNORE,	"IGNORE", parse_attribute_ignore},
	{SATTR_FORMAT,	"FORMAT", parse_attribute_format}, 
	{SATTR_VPORT,	"VPORT",  NULL},
	{SATTR_PROBE,	"PROBE",  parse_attribute_probe},
	{0, NULL, NULL},
} ;

///////////////////////////////////////////////////////////////////////////
// Print a formatted error message and maybe die.
//
// On entry:
//   die = 0 to return, 1 to exit(2)
//   fmt = printf-style format string 
//   ... arguments to print
//////////////////////////////////////////////////////////////////////////
void infile_error(int die, char *fmt, ...)
{
	fprintf(stderr, "\"%s\",%d: ", InFileName, InFileLineCnt);
	va_list ap;
	va_start(ap, fmt);
	vfprintf(stderr, fmt, ap);
	va_end(ap);
	putc('\n', stderr);
	if (die)
		exit(2);
}


///////////////////////////////////////////////////////////////////////////
// Read a line.  Store a copy for error messages. 
//
// On exit:
//   line      = string containing line
//   InLine    = contains safe copy of line (for error messages)
//   InLineCnt = current line number
//   return value = 0 if okay, 1 if eof
//////////////////////////////////////////////////////////////////////////
static int read_line(char *line)
{
	++InFileLineCnt;

	if (fgets(InFileLine, MAX_LINE_LEN, InFile) == NULL)
		return(1);

	strcpy(line, InFileLine);
	return(0);
}

///////////////////////////////////////////////////////////////////////////
// estimate number of cycles of data in a .in file, based on
// the number of bits required to be on each line and the file size
//
// Must be called after parse_infile_header()
///////////////////////////////////////////////////////////////////////////
int get_infile_num_vectors(int outspresent)
{
	int llen;	// minimum line length for vector lines
	int ncycs;	// maximum number of vectors in file
	struct stat st;

	llen = InNumSigs + InNumFields;
	if(outspresent) 
		llen += OutNumSigs + OutNumFields-1;
	
	if(fstat(fileno(InFile), &st) < 0) {
		fprintf(stderr, "fstat(%s): %s\n", InFileName, strerror(errno));
		exit(2);
	}
	ncycs =  st.st_size / llen + 1;
	return ncycs;
}

///////////////////////////////////////////////////////////////////////////
// Recursive routine to expand names possibly containing iterators.
// On first call (whith name != NULL), initialize variables and 
// return first name.  On subsequent calls, return next names.
// 
// On entry:
//   name  = (first call) signal name possibly containing interators 
//           (subsequent calls) NULL
//   level = level at which this function has been called
// On exit:
//   oname =  flattened signal name
//   rtn value = 0 if more names to go, 1 if last name
//////////////////////////////////////////////////////////////////////////

static int next_sig_name(char *iname, char *oname, int level)
{
  // Stack for recursion
  static char prefix [MAX_ITERATORS][MAX_NAME_LEN]; // name up until iterator
  static int  first  [MAX_ITERATORS];		    // iterator start val
  static int  last   [MAX_ITERATORS];		    // iterator end val
  static int  curr   [MAX_ITERATORS];		    // iterator cur val
  static int  digits [MAX_ITERATORS];		    // min. # of digits in iterator
  static int  more   [MAX_ITERATORS];		    // further levels?

  char *ptr, *firstptr, *lastptr;
  int   bump;

  // Check for overflow
  if (level >= MAX_ITERATORS)
    infile_error(1, "Maximum number of iterators exceeded");

  //
  // If first call load variables
  // 
  if (iname != NULL)
    {
      // Make local copy of name
      strcpy(prefix[level], iname);
      
      // Check for iterator
      if ((ptr = strchr(prefix[level], '{')) != NULL)
	{
	  // Prefix ends at iterator
	  *ptr = '\0';

	  // Parse the first number
	  firstptr = ptr+1;
	  if ((ptr = strchr(firstptr, ':')) == NULL)
	    infile_error(1, "Syntax error in iterator");
	  *ptr = '\0';
	  digits[level] = strlen(firstptr);
	  first [level] = atoi(firstptr);

	  // Parse the last number
	  lastptr = ptr+1;
	  if ((ptr = strchr(lastptr, '}')) == NULL)
	    infile_error(1, "Syntax error in iterator");
	  *ptr = '\0';
	  if ((int)strlen(lastptr) < digits[level])
	    digits[level] = strlen(lastptr);
	  last[level] = atoi(lastptr);

	  // Check if further calls needed
	  if (strlen(++ptr) > 0)
	    more[level] = 1;
	  else 
	    more[level] = 0;
	}
      else
	{
	  first [level]  = -1;
	  last  [level]  = -1;
	  more  [level]  = 0;
	}
      
      curr [level] = first[level];
    }

  // Add this level's contributation to oname
  strcat(oname, prefix[level]);
  if (curr[level] >= 0)
    {
      char iter[MAX_NAME_LEN];
      int  i;

      sprintf(iter, "%d", curr[level]);
      for (i = strlen(iter); i < digits[level]; i++)
	strcat(oname, "0");
      strcat(oname, iter);
    }

  // Get contribution from lower levels if necessary
  if (more[level])
    bump = next_sig_name((iname == NULL) ? NULL: ptr, oname, level+1);
  else
    bump = 1;    // always bump if this is the bottom level
  
  // Update iterator if necessary
  if (bump)
    {
      // Check if done
      if (curr[level] == last[level])
	{
	  curr[level] = first[level];
	  return(1);
	}

      if (first[level] < last[level])
	curr[level] += 1;
      else 
	curr[level] -= 1;
      return(0);
    }
  
  // More iterations to go
  return(0);

}

///////////////////////////////////////////////////////////////////////////
// next_name_alloc - get next name from iterator, allocating storage for
//	the name.
//	calls next_sig_name with a static buffer and then copies.
//
///////////////////////////////////////////////////////////////////////////
static int next_name_alloc(char *iname, char **oname, int level)
{
	char buf[MAX_NAME_LEN];
	buf[0] = 0;

	int done = next_sig_name(iname, buf, level);
	*oname = strdup(buf);
	if(!*oname)
		infile_error(1, "strdup: out of memory in next_name_alloc");
	return done;
}

///////////////////////////////////////////////////////////////////////////
// set parsing routine for an attribute
// If an attribute is only used by one client of the inparse library,
// the parsing routine is move there and the pointers set up here.
///////////////////////////////////////////////////////////////////////////
void set_attribute_parser(int id, ATTRCB afunc)
{
	int i;
	for(i = 0; AttrTab[i].a_name; i++) {
		if(id == AttrTab[i].a_id)
			AttrTab[i].a_func = afunc;
	}
}

///////////////////////////////////////////////////////////////////////////
// parse IGNORE attribute on signal field
///////////////////////////////////////////////////////////////////////////
static void parse_attribute_ignore(int id, char *value, Sig *sigt, Field *fld)
{
	if(strcasecmp(value, "NONE") == 0)
		sigt->ignore = IGNORE_NONE;
	else if (strcasecmp(value, "DELAY") == 0)
		sigt->ignore = IGNORE_DELAY;
	else if (strcasecmp(value, "SPICE") == 0)
		sigt->ignore = IGNORE_SPICE;
	else if (strcasecmp(value, "CAZM") == 0)
		sigt->ignore = IGNORE_SPICE;
	else if (strcasecmp(value, "ALL") == 0)
		sigt->ignore = IGNORE_ALL;
	else if (strcasecmp(value, "IRSIM") == 0)
		sigt->ignore = IGNORE_IRSIM;
	else
		infile_error(1, "garbled IGNORE attribute: %s", value);
}
///////////////////////////////////////////////////////////////////////////
// parse ETFO attribute on signal field
///////////////////////////////////////////////////////////////////////////
static void parse_attribute_etfo(int id, char *value, Sig *sigt, Field *fld)
{
	char *token;

	token = strtok(value, "+\n");
	if(!token) {
		infile_error(1, "syntax error in ETFO: fanout expected");
	}
	sigt->fanout = atof(token);
	token = strtok(NULL, ",\n");
	if(token)
		sigt->delay = atof(token);
	token = strtok(NULL, ",\n");
	if(token) {
		switch(toupper(*token)) {
		case 'F':
			sigt->edgestyle = EDGE_FALL;
			break;
		case 'R':
			sigt->edgestyle = EDGE_RISE;
			break;
		case 'B':
			sigt->edgestyle = EDGE_BOTH;
			break;
		default:
			infile_error(1, "edge style must be F, R, or B");
			break;
		}
	}
	token = strtok(NULL, " \t\n");
	if(token) {
		sigt->wavename = strdup(token);
	}
}
///////////////////////////////////////////////////////////////////////////
// parse FORMAT attribute on signal field
///////////////////////////////////////////////////////////////////////////
static void parse_attribute_format(int id, char *value, Sig *sigt, Field *fld)
{
	fld->leadzero = 0;  // default except for %b
	for( ; *value; value++) {
		switch(*value) {
		case '%':
			break;
		case '0':
			fld->leadzero = 1;
			break;
		case 'b':
		case 'd':
		case 'u':
		case 'o':
			fld->conv = *value;
			break;
		case 'h':
		case 'x':
			fld->conv = 'h';
			break;
		default:
			infile_error(1, "illegal format specfication");
			break;
		}
	}
}
///////////////////////////////////////////////////////////////////////////
// parse PROBE attribute on signal field
///////////////////////////////////////////////////////////////////////////
static void parse_attribute_probe(int id, char *value, Sig *sigt, Field *fld)
{
	fld->probestr = strdup(value);
}
///////////////////////////////////////////////////////////////////////////
// Parse signal attributes for either inputs or outputs section
// 
// On entry:
//   line = output line
// On exit:
//	Field pointed to by fld filled in with all required info
//		for the set of signals on this line
//	template Sig pointed to by sigt filled in with common information,
//		 ready to be copied to Sig's of actual signals.
//
// Side Effects:
//	line will be terminated with '\0' before attributes; only signals
//	will remain.
//	InFileActiveEdge updated.
//////////////////////////////////////////////////////////////////////////
static void parse_attributes(char *line, Sig *sigt, Field *fld)
{
	int i;
	char *token;
	char *cp;
	char *attrs;
	int found_attr;
	sigt->ignore = IGNORE_NONE;
	sigt->fanout = DEFAULT_FANOUT;
	sigt->delay = DEFAULT_DELAY;
	sigt->edgestyle = EDGE_FALL;
	sigt->wavename = NULL;
	sigt->name = NULL;

	fld->nsigs = 0;
	fld->leadzero = 1;
	fld->conv = 'b';
  
	// find attributes section, if any, by looking for '=' 
	// in first name=value attribute.
	cp = strchr(line, '=');
	if(cp) {
		// back up to beginning of first name=value pair by looking for
		// whitespace
		for(attrs = cp; !isspace(*attrs) && attrs > line; attrs--)
			;
		if(attrs <= line) {
			infile_error(1, "syntax error: signal name expected");
		}
		*attrs++ = 0;

		while(attrs) {
			// all this crud because strtok isn't reentrant
			char *pname = attrs;
			char *pval;
			pval = strchr(pname, '=');

			if(!pval) {
				infile_error(1, "syntax error: parameter value expected");
			}
			*pval++ = 0;	// terminates pname
			attrs = strpbrk(pval, " \t\n");
			if(attrs) {
				*attrs++ = 0;
				while(*attrs && isspace(*attrs))
					attrs++;
				if(*attrs == '\0')
					attrs = NULL;
			}
			
			// pname is attribute name,
			// pval is attribute value
			found_attr = 0;
			for(i = 0; AttrTab[i].a_name; i++) {
				if(strcasecmp(pname, AttrTab[i].a_name) == 0) {
					found_attr = 1;
					if(AttrTab[i].a_func)
						(AttrTab[i].a_func)(i, pval, sigt, fld);
				}
			}
			if(!found_attr) {
				infile_error(1, "unknown attribute %s=%s", pname, pval);			
			}
		} /* end while(attrs) */
		InfileActiveEdge |= sigt->edgestyle;
	} /* end if(attrs section) */
}

///////////////////////////////////////////////////////////////////////////
// post-parsing setup required for a Field structure
// - computes the field's column width based on number of bits and
//   format.
// 
///////////////////////////////////////////////////////////////////////////


static void field_ppsetup(Field *fld)
{

	// number of number of digits to display n-bit decimal number, 0 <= n <= 64
	static short ltab[] = {
		0,1,1,1,2,2,2,3,3,3,4,4,4,4,5,5,5,6,6,6,7,7,7,7,
		8,8,8,9,9,9,10,10,10,10,11,11,11,12,12,12,
		13,13,13,13,14,14,14,15,15,15,16,16,16,16,
		17,17,17,18,18,18,19,19,19,19,20 };

	switch(fld->conv) {
		case 'b':
			fld->width = fld->nsigs;
			break;
		case 'd':
		case 'u':
			if(fld->nsigs > 64) {
				infile_error(1, "Decimal output format not implemented for busses wider than 32 bits\n");
			}
			fld->width = ltab[fld->nsigs];
			break;
		case 'o':
			fld->width = (fld->nsigs-1)/3+1;
			break;
		case 'h':
			fld->width = (fld->nsigs-1)/4+1;
			break;
		default:
			infile_error(1, "Internal error fld->conv=%c\n", fld->conv);
			break;
	}
}

///////////////////////////////////////////////////////////////////////////
// Parse a line in the static inputs section
// 
// On entry:
//   line = input line
//////////////////////////////////////////////////////////////////////////
static void parse_static_line(char *line)
{
  char *token;
  int   nsigs;
  int   i;

  // Remove comments
  token = strchr(line, '#');
  if (token != NULL)
    *token = '\0';
  
  // Check for blank line
  token = strtok(line, " \t\n");
  if (token == NULL)
    return;
  
  // Process each signal name field
  nsigs = 0;
  do
    {
      int firstcall;
      int done;
      
      // Store instances of signal name
      firstcall = 1;
      do
	{
	  done = next_name_alloc((firstcall == 1) ? token : NULL, 
				 &(StatSigs[StatNumSigs].name), 0);
	  nsigs++;
	  StatNumSigs++;

	  if (StatNumSigs >= MAX_SIGS)
	    infile_error(1, "max number of static signals exceeded");

	  firstcall = 0;

	} while (!done);
      
      // Get next field
      if ((token = strtok(NULL, " \t\n")) == NULL)
	infile_error(1, "bit vector not found for static signals");
      
    } while (strspn(token, "01") < strlen(token));
  
  // Make sure bit vector has same length as number of signals on line
  if (nsigs != (int)strlen(token))
    infile_error(1, "wrong vector length for static signal");
  
  // Record bit values with signal names
  for (i = 0; i < (int)strlen(token); i++)
    {
      if      (token[i] == '0')
	StatSigs[StatNumSigs-nsigs+i].val  = 0;
      else if (token[i] == '1')
	StatSigs[StatNumSigs-nsigs+i].val  = 1;
      else
	infile_error(1, "static signal format error");
    }
}


///////////////////////////////////////////////////////////////////////////
// Parse a line in a signal spec section (input, output, monitor, etc).
// 
// On entry:
//   char *line = line read from .in file
//   char *ssname = name of signal set being read, for use in messages
//   SigSet *ss = pointer to structure containing Sigs and Fields arrays 
//////////////////////////////////////////////////////////////////////////
static void parse_sigspec_line(char *line, char *ssname, SigSet *ss)
{
	char *token;
  
	// Remove comments
	token = strchr(line, '#');
	if (token != NULL)
		*token = '\0';

	Sig sigtmp;
	parse_attributes(line, &sigtmp, &ss->Fields[ss->nFields]);

	token = strtok(line, " \t\n");
	if (token == NULL)
		return;
  
	// Process each signal name field
	ss->Fields[ss->nFields].name = strdup(token);
	ss->Fields[ss->nFields].firstsig = ss->nSigs;
	ss->Fields[ss->nFields].nsigs    = 0;

	do {
		int firstcall;
		int done;

		// Store instances of signal name
		firstcall = 1;
		do {
			ss->Sigs[ss->nSigs] = sigtmp; // copy attributes

			done = next_name_alloc((firstcall == 1) ? token : NULL, 
					       &(ss->Sigs[ss->nSigs].name), 0);

			ss->Fields[ss->nFields].nsigs++;
			ss->nSigs++;

			if (ss->nSigs >= MAX_SIGS || 
			    ss->Fields[ss->nFields].nsigs >= MAX_FIELD_SIGS)
				infile_error(1, "max number of %s signals exceeded", ssname);

			firstcall = 0;

		} while (!done);
      
	} while ((token = strtok(NULL, " \t\n")) != NULL);

	field_ppsetup(&ss->Fields[ss->nFields]);

	// Advance field counter
	ss->nFields++;
	if (ss->nFields >= MAX_FIELDS)
		infile_error(1, "max number of %s fields exceeded", ssname);
}


///////////////////////////////////////////////////////////////////////////
// Routine to open .in file and parse header
// 
// On entry:
//   name = name of .in file
// On exit:
//   Signal and field variables and arrays are filled in
//   Exits with error message if any errors encountered
//   otherwise leaves input file open and positioned to start parsing vectors
// 
//////////////////////////////////////////////////////////////////////////


void parse_infile_header(char *name)
{
	static char line[MAX_LINE_LEN];
	enum sectEnum {none=0, statics=1, inputs=2, outputs=4, monitor=8} cursect;
	regexp *secthead;
	char *headname;
	int seensects;
	int done = 0;

  // Initialize counters
	StatNumSigs   = 0;
	ISigs.nSigs   = 0;
	ISigs.nFields = 0;
	OSigs.nSigs   = 0;
	OSigs.nFields = 0;
	MonSigs.nSigs = 0;
	MonSigs.nFields = 0;

	secthead = regcomp("^([a-zA-Z]+):[ \t\n]$");  // will exit if error
	cursect = none;
	seensects = 0;

	// Open input file
	strcpy(InFileName, name);
	if ((InFile = fopen(InFileName, "r")) == NULL)
		infile_error(1, "opening .in file %s: %s", InFileName, strerror(errno));

// Read and parse header sections
// sections start with a line containing "keyword:"
	while(!done) {
	  // Read next line, looking for section keyword
		if(read_line(line))
			infile_error(1, "unexpected EOF eof in header");

		if(regexec(secthead, line) == 0) { 
			switch(cursect) { // add line to current section, if any
			case statics:
				parse_static_line(line);
				break;
			case inputs:
				parse_sigspec_line(line, "input", &ISigs);
				break;
			case outputs:
				parse_sigspec_line(line, "output", &OSigs);
				break;
			case monitor:
				parse_sigspec_line(line, "monitor", &MonSigs);
				break;
			default:
				break;
			}
		} else { // line matches section name pattern
			*secthead->endp[1] = 0;
			headname = secthead->startp[1];
			if (strcasecmp(headname, "static") == 0) {
				cursect = statics;
			} else if (strcasecmp(headname, "inputs") == 0) {
				cursect = inputs;
			} else if (strcasecmp(headname, "outputs") == 0) {
				cursect = outputs;
			} else if (strcasecmp(headname, "monitor") == 0) {
				cursect = monitor;
			} else if (strcasecmp(headname, "vectors") == 0) {
				done = 1;
			} else {
				infile_error(1, "unknown section header \"%s:\"", headname);
			}
			seensects |= cursect;
		}
	}
	if(InfileActiveEdge == EDGE_DEFAULT)
		InfileActiveEdge = EDGE_FALL;

	if((seensects & (inputs|outputs)) != (inputs|outputs)) {
		infile_error(1, ".in file needs to have at least one input and one output");
	}
}

  
///////////////////////////////////////////////////////////////////////////
// Routine to parse next input vector from .in file
// 
// On exit:
//   inflag  = set if input vectors found
//   invec   = output vector array
//   outflag = set if output vectors found
//   outvec  = output vector array
//   edgep:  if non-NULL, set to vector edge style, or 0 if none specified.
//   comment: if non-NULL, set to pointer into line to start of comment
//
//   Return value = 0 if okay
//                  1 if EOF
//                 -1 if non-fatal format error
//////////////////////////////////////////////////////////////////////////
int parse_infile_vector(int *inflag,  char *invec,
			int *outflag, char *outvec,
			char *line,
			int *edgep, char **comment)
{
	static char rdline[MAX_LINE_LEN];
	char  *ptr;
	char  *token;
	int    i;
	
	// Set defaults
	*inflag  = 0;
	*outflag = 0;
	
	// Read line, check for EOF
	if (read_line(rdline))
		return(1);
	
	// Copy rdline to line (output variable) before we trash it
	strcpy(line, rdline);
	
	// Remove comment if any
	if ((ptr = strchr(rdline, '#')) != NULL) {
		*ptr++ = '\0';
		if(comment) {
			while(isspace(*ptr))
				ptr++;
			*comment = ptr;
			ptr = strchr(ptr, '\n');
			if(ptr)
				*ptr = '\0';
		}
	} else {
		if(comment)
		*comment = NULL;
	}

	// If no input vectors, return now
	token = strtok(rdline, " \t\n");
	if (token == NULL)
		return(0);
  
	// Read input vectors
	invec[0] = '\0';
	for (i = 0; i < ISigs.nFields; i++) {
		if (token == NULL)
			infile_error(1, "input vector has wrong number of fields");

		touppers(token);
		// Copy field into return variable
		int nbits = strspn(token, "01XZ-");
		if (nbits != ISigs.Fields[i].nsigs)
		infile_error(1, "input vector field %d: found %d bits expected %d",
			     i, nbits, ISigs.Fields[i].nsigs);

		strcat(invec, token);
      
		// Parse next field
		token = strtok(NULL, " \t\n");
	}  
	*inflag = 1;

	if (token && strspn(token, "01XxZz-")>0 ) { // Output vectors present
		*outflag = 1;
		outvec[0] = '\0';
		for (i = 0; i < OSigs.nFields; i++) {
			if (token == NULL) {
				infile_error(0, "output vector has wrong number of fields");
				return(-1);
			}

			// Copy field into return variable, change 'x's to 'X's
			touppers(token);
			int l = strspn(token, "01XZ-");
			if (l != OSigs.Fields[i].nsigs) {
				infile_error(0, "output vector field %d: found %d bits expected %d",
					     i, l, OSigs.Fields[i].nsigs);
				return(-1);
			}

			strcat(outvec, token);
      
			// Parse next field
			token = strtok(NULL, " \t\n");
		}  
	} else {
		// no outvectors present; set outvec to all 'X's
		for (i = 0; i < OSigs.nSigs; i++)
			outvec[i] = 'X';
		outvec[OSigs.nSigs] = '\0';
	}

	int vecedge = 0;
	if (token != NULL) {
		switch(toupper(*token)) {
		case 'F':
			vecedge = EDGE_FALL;
			break;
		case 'R':
			vecedge = EDGE_RISE;
			break;
		default:
			infile_error(1, "vector edge indicator must be F or R");
			break;
		}
	}
	if(edgep)
		*edgep = vecedge;

	return(0);
}

/////////////////////////////////////////////////////////////////////////////
// tolowers - convert a string to lower case.  Nonalphabetics are unaffected.
/////////////////////////////////////////////////////////////////////////////
void tolowers(char *s)
{
	while(*s) {
		*s = tolower(*s);
		s++;
	}
}

/////////////////////////////////////////////////////////////////////////////
// touppers - convert a string to upper case.  Nonalphabetics are unaffected.
/////////////////////////////////////////////////////////////////////////////
void touppers(char *s)
{
	while(*s) {
		*s = toupper(*s);
		s++;
	}
}
