///////////////////////////////////////////////////////////////////////////
//
// ir2out -- Translate irsim output to vector-oriented .out format
//		.in file is used to determine edges at which to sample,
//		and to find signal names for printing header.
//
// Returns 0 if successful
//         1 if error occurred
//
// Written 13-Dec-94 by Steve Tell
//
// $Log: ir2out.C,v $
// Revision 1.9  2003/09/03 19:53:27  cvs
// bump revision number to 20030903
// improve comment output in incat
// add support for M= parameter to hspice-monte
// stuff for compilation on latest gcc
//
// Revision 1.8  1998/09/24 18:32:29  tell
// minor fixes to sync up with changes to inparse.h
//
// Revision 1.7  1998/09/24 17:50:01  tell
// Add support for "monitor" signals
//
//
//////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <stdarg.h>
#include <ctype.h>
#include <float.h>
#include <time.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <assert.h>
#include "inparse.h"
#include "vec.h"

//
// Constants
//

static int g_debug;
static int write_monsigs = 1;
static char *progname = "ir2out";
char edge_chars[] = "?FRB";

//
// Globals
//

// Active edge of clock
int ActiveEdge;

// Irsim file information
static FILE *IrFile;
static char *IrFileName;
static char  IrFileLine[MAX_LINE_LEN];
static int   IrFileLineCnt = 0;


///////////////////////////////////////////////////////////////////////////
// Print out usage info
//////////////////////////////////////////////////////////////////////////
static void usage()
{
	fprintf(stderr, "Usage:  ir2out [options] <infile> <irsimfile>\n");
	fprintf(stderr, "        <infile>     is a doirsim .in file\n");
	fprintf(stderr, "        <irsimfile>   is the file containing irsim output\n");
	fprintf(stderr, "  options:\n");
	fprintf(stderr, "        -m             don't include monitored signals in output\n");
	fprintf(stderr, "        -N             Vectors should not include input signals\n");
	fprintf(stderr, "        -w             Parse viewlogic viewsim output format\n");
	exit(3);
}


///////////////////////////////////////////////////////////////////////////
// Print a formatted error message and maybe die.
//
// On entry:
//   die = 0 to return, 1 to exit(1)
//   msg = message to print
//////////////////////////////////////////////////////////////////////////
static void irfile_error(int die, char *fmt, ...)
{
	fprintf(stderr, "%s: \"%s\", line %d: ", progname,
		IrFileName, IrFileLineCnt);
	va_list ap;
	va_start(ap, fmt);
	vfprintf(stderr, fmt, ap);
	va_end(ap);

	putc('\n', stderr);
	if (die)
		exit(1);
}

///////////////////////////////////////////////////////////////////////////
// Read a line from Irsim file.  Store a copy for error messages. 
//
// On exit:
//   line      = string containing line
//   InLine    = contains safe copy of line (for error messages)
//   InLineCnt = current line number
//   return value = 0 if okay, 1 if eof
//////////////////////////////////////////////////////////////////////////
static int read_irfile_line(char *line)
{
	++IrFileLineCnt;

	if (fgets(IrFileLine, MAX_LINE_LEN, IrFile) == NULL)
		return(1);

	strcpy(line, IrFileLine);
		return(0);
}


// type for pointer to a parse_*_vector function

typedef int (*PVF)(char *invec, char *outvec, char *monvec, char *clockp, char *commp);

///////////////////////////////////////////////////////////////////////////
// Read and parse an input/output vector from the irsim raw output file
//
// On exit:
//   invec: vector of current state of input signals
//   outvec: vector of current state of input signals
//   clock: filled in with last state of clock signal, either 0, 1, or X.
//   commp: if non-NULL, points to char[]
//	filled in with null-terminated comment
//   
//   return value = 0 if okay, 1 if eof
//
// Parsing strategy:
//  lines beginning with the strings "__clock__=" "__invec_" and "__outvec_" build up
//  the data for a vector.
//  A line beginning with "time" ends a vector.
//  other lines are ignored.
//  If a vector contains no data at all (just at "time" line) it is skipped.
//
//////////////////////////////////////////////////////////////////////////
int 
parse_irsim_vector(char *invec, char *outvec, char *monvec, char *clockp, char *commp)
{
	char lbuf[MAX_LINE_LEN];
	char *cp, *ep;
	int got_data = 0;

	// initialize
	*clockp = 'X';
	memset(invec, InNumSigs, '-');
	invec[InNumSigs] = '\0';
	memset(outvec, OutNumSigs, '-');
	outvec[OutNumSigs] = '\0';
	memset(monvec, MonSigs.nSigs, '-');
	monvec[MonSigs.nSigs] = '\0';
	commp[0] = '\0';

	while(read_irfile_line(lbuf) != 1) {
		if(got_data && strncmp(lbuf, "time", 4) == 0) {
			return 0;
		} else if(strncmp(lbuf, "__clock__=", 10)==0 ||
		   strncmp(lbuf, "__invec_", 8)==0 ||
		   strncmp(lbuf, "__monvec_", 9)==0 ||
		   strncmp(lbuf, "__outvec_", 9)==0) {
			char *token;
			int fld;
			char *fdat;
			token = strtok(lbuf, " \t\n");
			while(token != NULL) {
				got_data = 1;
				if(strncmp(token, "__clock__=", 10)==0) {
					*clockp = token[10];
				} else if(strncmp(token, "__invec_", 8)==0) {
					fld = atoi(&token[8]);
					if(fld==0)
						irfile_error(1,"field number expected, got \"%s\"",token);
					fld--;
					fdat = strchr(token, '=');
					if(!fdat)
						irfile_error(1,"= expected, got\"%s\"",token);
					fdat++;
					if(strlen(fdat) != InFields[fld].nsigs)
						irfile_error(1,"wrong number of bits in __invec_%d for field %s",
							fld, InFields[fld].name);
					memcpy(&invec[InFields[fld].firstsig],
					       fdat, InFields[fld].nsigs);
				} else if(strncmp(token, "__outvec_", 9)==0) {
					fld = atoi(&token[9]);
					if(fld==0)
						irfile_error(1,"field number expected, got \"%s\"",token);
					fld--;
					fdat = strchr(token, '=');
					if(!fdat)
						irfile_error(1,"= expected, got\"%s\"",token);
					fdat++;
					if(strlen(fdat) != OutFields[fld].nsigs)
						irfile_error(1,"wrong number of bits in __outvec_%d for field %s",
							fld, OutFields[fld].name);
					memcpy(&outvec[OutFields[fld].firstsig],
					       fdat, OutFields[fld].nsigs);
				} else if(strncmp(token, "__monvec_", 9)==0) {
					fld = atoi(&token[9]);
					if(fld==0)
						irfile_error(1,"field number expected, got \"%s\"",token);
					fld--;
					fdat = strchr(token, '=');
					if(!fdat)
						irfile_error(1,"= expected, got\"%s\"",token);
					fdat++;
					if(strlen(fdat) != MonSigs.Fields[fld].nsigs)
						irfile_error(1,"wrong number of bits in __monvec_%d for field %s",
							fld, MonSigs.Fields[fld].name);
					memcpy(&monvec[MonSigs.Fields[fld].firstsig],
					       fdat, MonSigs.Fields[fld].nsigs);
				}

				token = strtok(NULL, " \t\n");
			}
		} else if((cp = strchr(lbuf, '|')) != NULL) {
			if((ep = strchr(cp, '\n')))
				*ep = '\0';
			cp++;
			if(*cp != '#')
				printf("#%s\n", cp);
			else if(commp)
				strcpy(commp, cp);
		}
	}

	return EOF;
}

///////////////////////////////////////////////////////////////////////////
// Read and parse an input/output vector from the raw output file
//	This version reads viewsim's gratuitiously different output format.
//
// On exit:
//   invec: vector of current state of input signals
//   outvec: vector of current state of input signals
//   clock: filled in with last state of clock signal, either 0, 1, or X.
//   commp: if non-NULL, points to char[]
//	filled in with null-terminated comment
//   
//   return value = 0 if okay, 1 if eof
//
// Parsing strategy:
//  lines beginning with the strings "time", "__clock__=", "__invec_",
//  and "__outvec_", build up the data for a vector.  
//  The token sequence "time = <whatever>" is skipped.
//
//  A line beginning with "simulation" ends a vector.
//  other lines are ignored.
//  If a vector contains no data at all (just a "time" line) it is skipped.
//
//////////////////////////////////////////////////////////////////////////
int 
parse_viewsim_vector(char *invec, char *outvec, char *monvec, char *clockp, char *commp)
{
	char lbuf[MAX_LINE_LEN];
	char *lp, *cp, *ep;
	int got_data = 0;

	// initialize
	*clockp = 'X';
	memset(invec, InNumSigs, '-');
	invec[InNumSigs] = '\0';
	memset(outvec, OutNumSigs, '-');
	outvec[OutNumSigs] = '\0';
	memset(monvec, MonSigs.nSigs, '-');
	monvec[MonSigs.nSigs] = '\0';
	commp[0] = '\0';

	while(read_irfile_line(lbuf) != 1) {
		tolowers(lbuf);		// fold to lower case
		lp = &lbuf[0];
		while(*lp && (*lp == '|' || isspace(*lp)))
			lp++;

		if(got_data && strncmp(lp, "simulation", 10) == 0) {
			return 0;
		} else if(strncmp(lp, "__clock__=", 10)==0 ||
		   strncmp(lp, "time", 4)==0 ||
		   strncmp(lp, "__invec_", 8)==0 ||
		   strncmp(lp, "__monvec_", 9)==0 ||
		   strncmp(lp, "__outvec_", 9)==0) {
			char *token;
			int fld;
			char *fdat;
			token = strtok(lp, " \t\n");
			while(token != NULL) {
				got_data = 1;
				if(strncmp(token, "time", 4)==0) {
					/* discard '=' and number. */
					(void) strtok(NULL, " \t\n");
					(void) strtok(NULL, " \t\n");

				} else if(strncmp(token, "__clock__=", 10)==0) {
					*clockp = token[10];

				} else if(strncmp(token, "__invec_", 8)==0) {
					fld = atoi(&token[8]);
					if(fld==0)
						irfile_error(1,"field number expected, got \"%s\"",token);
					fld--;
					fdat = strchr(token, '=');
					if(!fdat)
						irfile_error(1,"= expected, got\"%s\"",token);
					fdat++;
					if(strlen(fdat) != InFields[fld].nsigs)
						irfile_error(1,"wrong number of bits in __invec_%d for field %s",
							fld, InFields[fld].name);
					memcpy(&invec[InFields[fld].firstsig],
					       fdat, InFields[fld].nsigs);

				} else if(strncmp(token, "__outvec_", 9)==0) {
					fld = atoi(&token[9]);
					if(fld==0)
						irfile_error(1,"field number expected, got \"%s\"",token);
					fld--;
					fdat = strchr(token, '=');
					if(!fdat)
						irfile_error(1,"= expected, got\"%s\"",token);
					fdat++;
					if(strlen(fdat) != OutFields[fld].nsigs)
						irfile_error(1,"wrong number of bits in __outvec_%d for field %s",
							fld, OutFields[fld].name);
					memcpy(&outvec[OutFields[fld].firstsig],
					       fdat, OutFields[fld].nsigs);
				} else if(strncmp(token, "__monvec_", 9)==0) {
					fld = atoi(&token[9]);
					if(fld==0)
						irfile_error(1,"field number expected, got \"%s\"",token);
					fld--;
					fdat = strchr(token, '=');
					if(!fdat)
						irfile_error(1,"= expected, got\"%s\"",token);
					fdat++;
					if(strlen(fdat) != MonSigs.Fields[fld].nsigs)
						irfile_error(1,"wrong number of bits in __monvec_%d for field %s",
							fld, MonSigs.Fields[fld].name);
					memcpy(&monvec[MonSigs.Fields[fld].firstsig],
					       fdat, MonSigs.Fields[fld].nsigs);
				}

				token = strtok(NULL, " \t\n");
			}
		} else if((cp = strchr(lp, '|')) != NULL) {
			if((ep = strchr(cp, '\n')))
				*ep = '\0';
			cp++;
			if(*cp != '#')
				printf("#%s\n", cp);
			else if(commp)
				strcpy(commp, cp);
		}
	}

	return EOF;
}



///////////////////////////////////////////////////////////////////////////
// Main routine
//////////////////////////////////////////////////////////////////////////

int main(int argc, char **argv)
{
	int   i;
	int c;
	char *infile;
	extern char *optarg;
	extern int optind;
	extern int optopt;

	// Flags
	int inputs=1;	 	// CAzM waveforms and output files include input signals ?
	int edge_override = -1;
	char *ep;
	int viewLogic = 0;

	// parse arguments
	while((c = getopt(argc, argv, "Ne:mwx")) != -1) {
		switch(c) {
		case 'N':
			inputs= 0;
			break;
		case 'e':
			ep = strchr(edge_chars, optarg[0]);
			if(!ep) {
				fprintf(stderr, "%s: -e: edge indicator must be 'F', 'R', or 'B'\n", progname);
				exit(1);
			}
			edge_override = ep-edge_chars;
			break;
		case 'm':
			write_monsigs = 0;
			break;
		case 'w':
			viewLogic = 1;
			break;
		case 'x':
			g_debug = 1;
			break;
		default:
			fprintf(stderr, "%s: Unrecognized option: -%c\n", progname, optopt);
			usage();
			break;
		}
	}
	if(optind + 2 != argc) {
		fprintf(stderr, "%s: infile and irsimfile not specified\n", progname);
		usage();
	}

	infile = argv[optind];
	IrFileName = argv[optind+1];

	// Parse .in file header and set up signal sampling schedule
	parse_infile_header(infile);

	if(edge_override != -1)
		ActiveEdge = edge_override;
	else
		ActiveEdge = InfileActiveEdge;

	if(g_debug)
	   fprintf(stderr, "ActiveEdge=%c\n", edge_chars[ActiveEdge]);

	// Open irsim raw output file
	if ((IrFile = fopen(IrFileName, "r")) == NULL) {
		perror(IrFileName);
		exit(2);
	}

	struct stat statbuf;
	time_t timeval;
	if(fstat(fileno(IrFile), &statbuf) == -1) {
		struct timeval tp;
		struct timezone tz;
		gettimeofday(&tp, &tz);
		timeval = tp.tv_sec;
	} else {
		timeval = statbuf.st_mtime;
	}

	static char line      [MAX_LINE_LEN];
	sprintf(line, "IRSIM Simulation %s\n", ctime(&timeval));
	print_outfile_header(line, inputs, write_monsigs);

	static char invec     [MAX_SIGS];
	static char outvec    [MAX_SIGS];
	static char monvec    [MAX_SIGS];
	static char p_invec     [MAX_SIGS];
	static char p_outvec    [MAX_SIGS];
	static char p_monvec    [MAX_SIGS];
	int edge;
	int cycle = 0;
	int nedges = 0;
	char clock = 'X';
	char prev_clock = 'X';
	int nvec = 0;
	int init_done = 0;
	char comment[MAX_LINE_LEN];

	PVF parsefunc;
	if(viewLogic) {
		parsefunc = parse_viewsim_vector;
		if(g_debug)
			fprintf(stderr, "viewlogic mode\n");
	} else {
		parsefunc = parse_irsim_vector;
	}
		

	// "Sampling" stragegy: when clock changes, print the bit values
	// from the previous timestep (iteration).  This is because irsim
	// doesn't write the output lines until all events triggered by the
	// clock have occured (i.e. it doesn't understand propagation delays).
	// We want the values present when the clock changed, not after the
	// flip-flops have settled down.

	while( (parsefunc)(invec, outvec, monvec, &clock, comment) != EOF) {
		nvec++;
		if(clock == 'X')
			continue;

		if(nedges == 0 || clock != prev_clock) {
			switch(clock) {
				case '0':
					edge = EDGE_FALL;
					break;
				case '1':
					edge = EDGE_RISE;
					break;
				default:
					fprintf(stderr, "clock=%c\n", clock);
					abort();
			}

			nedges++;

			if(init_done && ((edge & ActiveEdge) != 0)) {

				if(ActiveEdge == EDGE_RISE ||
				   edge == EDGE_FALL)
					cycle++;
				for(i = 0; i < InNumSigs; i++) {
					if((InSigs[i].edgestyle & edge)==0)
						p_invec[i] = '-';
				}
				for(i = 0; i < OutNumSigs; i++) {
					if((OutSigs[i].edgestyle & edge)==0)
						p_outvec[i] = '-';
				}
				print_vector(stdout, p_invec, p_outvec, 
					     write_monsigs ? p_monvec : NULL,
					     inputs, 0, 
					     "  %c %-4d#%s",
					     tolower(edge_chars[edge]), cycle,
					     comment);
			} 
			// skip first two edges: initialization steps.
			if(!init_done) {
				if(edge == EDGE_RISE && nvec > 1)
					init_done = 1;
			}

		}
		prev_clock = clock;
		memcpy(p_invec, invec, sizeof(invec));
		memcpy(p_outvec, outvec, sizeof(outvec));
		memcpy(p_monvec, monvec, sizeof(monvec));
	}

	if(g_debug)
		fprintf(stderr, "%d edges, %d vectors\n", nedges, nvec);
	exit(0);
}
