///////////////////////////////////////////////////////////////////////////
//
// outchk -- Compare test vectors in a .in file with outputs in a
//           .out file.  Print log of all vectors (with errors flagged)
//           to stdout.  Echo errors to stderr.
//
//           Return 0 if all test vectors matched, 1 if no test vectors
//           found, >=2 if error.
//
// Originaly written 6/28/94 by Steve Molnar
// A number of overhauls along the way by Steve Tell.
//
//////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <stdarg.h>
#include <ctype.h>
#include "inparse.h"
#include "vec.h"

static char *progname = "outchk";

// Input file information
static FILE *OutFile;
static char  OutFileName[MAX_NAME_LEN];
static char  OutFileLine[MAX_LINE_LEN];
static int   OutFileLineCnt = 0;

static char edge_chars[] = "?frb"; /* order must be same as defines in inparse.h */

///////////////////////////////////////////////////////////////////////////
// Print an error message and maybe die.
//
// On entry:
//   msg = message to print
//   die = 0 to return, 1 to exit(2)
//////////////////////////////////////////////////////////////////////////
void outfile_error(int die, char *fmt, ...)
{
	fprintf(stderr, "%s: \"%s\",%d: ", progname, 
		OutFileName, OutFileLineCnt);
	va_list ap;
	va_start(ap, fmt);
	vfprintf(stderr, fmt, ap);
	va_end(ap);
	putc('\n', stderr);
	if (die)
		exit(2);
}

///////////////////////////////////////////////////////////////////////////
// Read a line from OutFile.  Store a copy for error messages. 
//
// On exit:
//   line      = string containing line
//   InLine    = contains safe copy of line (for error messages)
//   InLineCnt = current line number
//   return value = 0 if okay, 1 if eof
//////////////////////////////////////////////////////////////////////////
static int read_outfile_line(char *line)
{
  ++OutFileLineCnt;

  if (fgets(OutFileLine, MAX_LINE_LEN, OutFile) == NULL)
    return(1);

  strcpy(line, OutFileLine);
  return(0);
}


///////////////////////////////////////////////////////////////////////////
// Parse one cycle's worth of information from .out file.
// Report vector of output signal values in outvec and (spice/CAzM runs only)
// delay information in delayvec.
//
// On entry:
//   cazmflag = 1 if parsing CAzM outfile (has delay information)
//
// On exit:
//   vecflag  = 0 if line contains input vectors, otherwise OR of
//		1 = inputs present, 2 = outputs, 4 = monitors present
//
//   invec    = array consisting of [01XZ-]  for each input signal
//   outvec   = array consisting of [01XZ-] for each output signal
//   delayvec = array consisting of [01]   for each output signal
//              '0' means timing OK
//              '1' means violated setup time
//   monvec   = array consisting of [01XZ] for each monitored signal
//
//   line     = actual line from file
//   edgep    = clock edge indicator from vector
//   cycle    = cycle number, if found.  Else -1.
//   comment  = pointer into line to start of comment.
//
//   Return value = 0 if okay
//                  1 if EOF
//////////////////////////////////////////////////////////////////////////
int parse_outfile_vector(int cazmflag, int *vecflag, char *invec, 
			 char *outvec, char *delayvec, char *monvec, char *line,
			 int inputs, int *edgep, int *cycle, char **comment)
{
	static char rdline[MAX_LINE_LEN];
	char *ptr;
	char *token;
	int   i, j;
	
	*vecflag  = 0;
	// Read line, check for EOF
	if (read_outfile_line(rdline))
	return(1);
	
	// Copy rdline to line (output variable) before we trash it
	strcpy(line, rdline);
	
	// Remove comment if any
	if ((ptr = strchr(rdline, '#')) != NULL) {
		*ptr++ = '\0';
		if(comment) {
			*comment = ptr;
			ptr = strchr(ptr, '\n');
			if(ptr)
			*ptr = '\0';
		}
	} else {
		if(comment)
		*comment = NULL;
	}
	
	// If no vectors, return now
	token = strtok(rdline, " \t\n");
	if (token == NULL)
	{
		return(0);
	}
	
	// Read input vectors
	strcpy(invec, "");
	if (inputs)
		for (i = 0; i < InNumFields; i++) {
			if (token == NULL)
				infile_error(1, "input vector has wrong number of fields");

			// Copy field into return variable
			touppers(token);
			if (strspn(token, "01XZ-") != InFields[i].nsigs)
				outfile_error(1, "bad input vector");
			
			strcat(invec, token);
			*vecflag |= 1;
			// Parse next field
			token = strtok(NULL, " \t\n");
		}  
	
	int firstout_index = token - rdline;

	// Read output vectors
	outvec[0] = '\0';
	for (i = 0; i < OSigs.nFields; i++)
	{
		if (token == NULL)
			outfile_error(1, "output vector has too few fields; expected %d", OSigs.nFields);
		
		// Copy field into return variable, change 'x's to 'X's
		touppers(token);
		if (strspn(token, "01XZ-") != OutFields[i].nsigs)
			outfile_error(1, "bad characters in output vector field %d", i);
		
		strcat(outvec, token);
		*vecflag |= 2;
	
		// Parse next field
		token = strtok(NULL, " \t\n");
	}  

	// Read monitor vectors
	monvec[0] = '\0';
	for (i = 0; i < MonSigs.nFields; i++)
	{
		if (token == NULL)
			outfile_error(1, "monitor vector has too few fields; expected %d", MonSigs.nFields);
		
		// Copy field into return variable, change 'x's to 'X's
		touppers(token);
		if (strspn(token, "FR"))  // looks like an edge indicator - bail out of monitors
			break;  
		if (strspn(token, "01XZ-") != MonSigs.Fields[i].nsigs)
			outfile_error(1, "bad characters in monitor vector");
		
		strcat(monvec, token);
		*vecflag |= 4;

		// Parse next field
		token = strtok(NULL, " \t\n");
	}  
	
	int vecedge = 0;
	if (token != NULL) {
		switch(toupper(*token)) {
		case 'F':
			vecedge = EDGE_FALL;
			break;
		case 'R':
			vecedge = EDGE_RISE;
			break;
		default:
			infile_error(1, "vector edge indicator must be F or R");
			break;
		}
	}
	if(edgep)
		*edgep = vecedge;
	
	token = strtok(NULL, " \t\n");
	if(cycle) {
		if(token)
			*cycle = atoi(token);
		else
			*cycle = -1;
	}
	
	// If cazmflag set, read in delay vector
	if (cazmflag)
	{
		// Read line, check for EOF
		if (read_outfile_line(rdline))
			return(1);
		
		if (strlen(rdline) < inputs*InNumSigs+inputs*InNumFields+OutNumSigs+OutNumFields-1)
			outfile_error(1, "truncated delay vector");
		
		for (i = 0; i < OutNumFields; i++)
			for (j = 0; j < OutFields[i].nsigs; j++) {
				int srcindex;
				int dstindex;
			
				srcindex = firstout_index + 
						OutFields[i].firstsig+i+j;
				dstindex = OutFields[i].firstsig+j;
			
				switch(rdline[srcindex]) {
				case ' ':
					delayvec[dstindex] = ' ';
					break;
				case '*':
					delayvec[dstindex] = 's';
					break;
				default:
					if(isalpha(rdline[srcindex]))
						delayvec[dstindex] = tolower(rdline[srcindex]);
					else
						outfile_error(1, "garbled delay vector");
					break;
				}
			}
		// Sanity check
		for (i = 0; i < OutNumSigs; i++)
			if (delayvec[i] != ' ' && !islower(delayvec[i]))
				outfile_error(1, "garbage characters in delay vector");
	}
	
	return(0);
}

///////////////////////////////////////////////////////////////////////////
// Compare vectors from each source and print out an error report if
// necessary.
// 
// On entry:
//   cazmflag        = 1 if cazm run, 0 otherwise
//   hlflag          = 1   if highlighted format instead of multi-line 
//		     error output is desired.
//   incycle         = cycle number according to .in file
//   outcycle        = cycle number according to .out file
//   invec,  outvec  = in,out vectors from .in file
//   sinvec, soutvec = in,out vectors from .out file
//   smonvec         = monitor vector from .out file - simply printed out
//   inedge, outedge = edge indicators for .in, .out file
//   delayvec        = timing error vector for output signals
//   dobase	     = 0 forces all output formats to binary
//   incom, outcom   = in and out file comments, less #
//
// On exit:
//   rtn value = 0 if vectors match
//              -1 if mismatch
//////////////////////////////////////////////////////////////////////////
int process_vectors(int cazmflag, int hlflag, int incycle, int outcycle,
		    char *invec, char *outvec,
		    char *sinvec, char *soutvec, char *smonvec,
		    int inedge, int outedge,
		    char *delayvec, int inputs, int dobase,
		    char *incom, char *outcom)
{
	static char inerrvec [MAX_SIGS];       // Error vectors
	static char outerrvec[MAX_SIGS];    
	int errfnd;
	int i;
	int cycle_err = 0;
	int edge_err = 0;
	
	//
	// Compare vectors from two sources
	//
	errfnd  = 0;
	
	// Check cycle number and type for consistency
	if (incycle != outcycle) {
		errfnd = 1;
		cycle_err = 1;
	}
	if (inedge != outedge) {
		errfnd = 1;
		edge_err = 1;
	}
	
	// Check inputs (as a sanity check)
	// but we do obey the appropriate IGNORE flag
	if (inputs)
	for (i = 0; i < InNumSigs; i++) {
		if(cazmflag && (InSigs[i].ignore & IGNORE_SPICE)
		   || !cazmflag && (InSigs[i].ignore & IGNORE_IRSIM))
			inerrvec[i] = ' ';
		else {
			if((invec[i] == sinvec[i])
			   || (invec[i] == '-')
			   || (sinvec[i] == '-')
			   || (invec[i] == 'X')
			   || (invec[i] == 'Z') ) {
				inerrvec[i] = ' ';
			} else {
				inerrvec[i] = '^';
				errfnd = 1;
			}
		}
	}
	
	// Now the outputs
	for (i = 0; i < OutNumSigs; i++) {
		if (cazmflag) {
			if ((OutSigs[i].ignore == IGNORE_SPICE) || 
			    (OutSigs[i].ignore == IGNORE_ALL)  ||
			    (OutSigs[i].ignore == IGNORE_DELAY && outvec[i]  == soutvec[i]) ||
			    (outvec[i] == 'X')                 ||
			    (outvec[i] == soutvec[i] && delayvec[i] == ' ') ||
			    (outvec[i] == '-') ||
			    (soutvec[i] == '-')) {
				outerrvec[i] = ' ';
			} else if (outvec[i] == soutvec[i] && delayvec[i] != ' ') {
				outerrvec[i] = delayvec[i];
				errfnd = 1;
			} else {
				outerrvec[i] = '^';
				errfnd = 1;
			}
		} else { // IRSIM--no delay vector
			if ((OutSigs[i].ignore == IGNORE_ALL)    ||
			    (OutSigs[i].ignore == IGNORE_IRSIM)  ||
			    (outvec[i] == 'X')                   ||
			    (outvec[i] == soutvec[i]) ||
			    (outvec[i] == '-') ||
			    (soutvec[i] == '-')) {
				outerrvec[i] = ' ';
			} else {
				outerrvec[i] = '^';
				errfnd = 1;
			}
		}  
	}
	
	//
	// If no error, print outfile vector and return
	//
	
	if (!errfnd) {
		print_vector(stdout, sinvec, soutvec, smonvec, inputs, dobase,
			     "  %c %-4d#%s %s",
			     edge_chars[outedge], outcycle, outcom?outcom:"",
			     incom?incom:"");
		return(0);
	}
	
	//
	// Print error report to stdout and stderr
	//
	
	// Error message to stderr
	outfile_error(0, "vector mismatch");

	if(incom==NULL)
		incom=" (expected)";
	if(hlflag) {
		print_vector(stderr, sinvec, soutvec, smonvec, inputs, dobase,
			     "  %c %-4d#%s",
			     edge_chars[outedge], outcycle, outcom?outcom:"");
		print_vector(stderr, invec, outvec, NULL, inputs, dobase,
			     "  %c %-4d#%s",
			     edge_chars[inedge], incycle, incom);

		// Error vector
		if (cazmflag) {
			print_err_vector(stderr, inerrvec, outerrvec, inputs, 1, dobase,
			     "  %c %s # s = setup, ^ = error",
			     edge_err?'^':' ', cycle_err?"^^^":"   ");
		} else {
			print_err_vector(stderr, inerrvec, outerrvec, inputs, 1, dobase,
			     "  %c %s # ^ = error",
			     edge_err?'^':' ', cycle_err?"^^^":"   ");
		}

		print_hl_vector(stdout, sinvec, soutvec, smonvec, inerrvec, outerrvec,
				inputs, dobase, "  %c %-4d#%s",
				edge_chars[outedge], outcycle,
				outcom?outcom:"");

	} else {
		// Separator line to stdout
		fprintf(stdout, "----\n");
	
		// Outfile vector
		print_vector(stdout, sinvec, soutvec, smonvec, inputs, dobase, "  %c %-4d#%s", edge_chars[outedge], outcycle, outcom?outcom:"");
		print_vector(stderr, sinvec, soutvec, smonvec, inputs, dobase, "  %c %-4d#%s", edge_chars[outedge], outcycle, outcom?outcom:"");
	
		// Infile vector
		print_vector(stdout, invec, outvec, NULL, inputs, dobase, "  %c %-4d#%s", edge_chars[inedge], incycle, incom);
		print_vector(stderr, invec, outvec, NULL, inputs, dobase, "  %c %-4d#%s", edge_chars[inedge], incycle, incom);
	
		// Error vector
		if (cazmflag) {
			print_err_vector(stdout, inerrvec, outerrvec, inputs, 1, dobase, 
			     "  %c %s # s = setup, ^ = error",
			     edge_err?'^':' ', cycle_err?"^^^":"   ");
			print_err_vector(stderr, inerrvec, outerrvec, inputs, 1, dobase,
			     "  %c %s # s = setup, ^ = error",
			     edge_err?'^':' ', cycle_err?"^^^":"   ");
		} else {
			print_err_vector(stdout, inerrvec, outerrvec, inputs, 1, dobase,
			     "  %c %s # ^ = error",
			     edge_err?'^':' ', cycle_err?"^^^":"   ");
			print_err_vector(stderr, inerrvec, outerrvec, inputs, 1, dobase,
			     "  %c %s # ^ = error",
			     edge_err?'^':' ', cycle_err?"^^^":"   ");
		}
	
		// Separator line to stdout
		fprintf(stdout, "----\n");
	}
	
	return(-1);
}

void usage()
{
	fprintf(stderr, "Usage: outchk [options] <infile> <outfile>\n");
	fprintf(stderr, " options:\n");
	fprintf(stderr, " -b show all outputs in binary instead of specified base.\n");
	fprintf(stderr, " -d assume outfile contains delay/setup vectors; else autodetect.\n");
	fprintf(stderr, " -h show errors with highlighting instead of multi-line got/expected format.\n");
	fprintf(stderr, " -m Do not expect monitor signals to be present.\n");
	fprintf(stderr, " -N input vectors are not present\n");
}

///////////////////////////////////////////////////////////////////////////
// Main routine
//////////////////////////////////////////////////////////////////////////

int main(int argc, char **argv)
{
	static char invec   [MAX_SIGS];	        // Variables for parsing .in file vectors
	static char outvec  [MAX_SIGS];
	int    inflag,  outflag;
	static char sinvec  [MAX_SIGS];	        // Variables for parsing .out file vectors
	static char soutvec [MAX_SIGS];
	static char delayvec[MAX_SIGS];
	static char smonvec [MAX_SIGS];
	int    vecflag;
	int	hlflag = 0;
	static char line    [MAX_LINE_LEN];		// Other variables
	int    cazmflag = 0;
	int    rtnval;
	int    incycle, outcycle;
	int    ineof,   outeof;
	char  *ptr;
	int inputs = 1;
	int dobase = 1;	// allow output conversions besides binary
	int expect_monsigs = 1;
	int c;
	int err = 0;
	extern char *optarg;
	extern int optind;
	extern int optopt;

	// parse arguments
	while((c = getopt(argc, argv, "bdhmN")) != -1) {
		switch(c) {
		case 'N':
			inputs= 0;
			break;
		case 'h':
			hlflag = 1;
			break;
		case 'd':
			cazmflag = 1;
			break;
		case 'm':
			expect_monsigs = 0;
			break;
		case 'b':
			dobase = 0;
			break;
		default:
			fprintf(stderr, "%s: Unrecognized option: -%c\n", progname, optopt);
			usage();
			err = 1;
			break;
		}
	}
	if(optind + 2 != argc) {
		fprintf(stderr, "%s: infile and outfile not specified\n", progname);
		err = 1;
		usage();
	}
	if(err)
		exit(2);

	char *infile = argv[optind];
	char *outfile = argv[optind+1];

	// Initialization
	ineof   = 0;
	outeof  = 0;
	rtnval  = 1;	// init to no test vectors found

	// Parse .in file header
	parse_infile_header(infile);

	// Open .out file
	strcpy(OutFileName, outfile);
	if ((OutFile = fopen(outfile, "r")) == NULL) {
		perror(outfile);
		exit(3);
	}

	// Read first line of outfile and set cazmflag
	if (read_outfile_line(line))
		outfile_error(1, "truncated file");
	
	if(!cazmflag) {
		if (strncasecmp(line, "cazm", 4) == 0)
			cazmflag = 1;
	}

	// assumes that cazm .out files don't have monitor bits, and 
	// irsim ones will.  If this ever changes, we should put more
	// format indications in all of the header to tell us what to do.
	// Write header to log (on stdout)
	print_outfile_header(line, inputs,
			     expect_monsigs,
			     dobase);

	// Skip through .out file down to test vectors
	do {
		if (read_outfile_line(line)) {
			outeof = 1;
			break;
		}
	} while ((line[0] != '-') && (line[0] != '='));

	// Compare vectors
	int in_edge;
	int out_edge;
	incycle = 0;
	char *in_comment;
	char *out_comment;
	while (!ineof && !outeof) {
		// Get next vector from .in file
		do {
			int status;

			status = parse_infile_vector(&inflag, invec, &outflag,
				       outvec, line, &in_edge, &in_comment);
			if (status ==  1) 
				ineof = 1;
			if (status == -1)
				infile_error(1, "format error, so cannot compare vectors");
	  
			// Write comment lines to stdout
			if (!inflag && !ineof)
				printf("%s", line);

			// If output vector present, update rtnval
			// (if no output vector, outvec will be all X's)
			if (outflag)
				if (rtnval == 1) rtnval = 0;

		} while (ineof == 0 && inflag == 0);

		if(InfileActiveEdge == EDGE_BOTH) {
			if(in_edge == EDGE_FALL || in_edge == 0)
				incycle++;
		} else {
			incycle++;
		}

		/* backward compatibility with .in files that don't specify edge. */
                if(InfileActiveEdge == EDGE_FALL && in_edge == 0)
			in_edge = EDGE_FALL;

		// Get next vector from .out file
		do {
			if (parse_outfile_vector(cazmflag, &vecflag,
						 sinvec, soutvec,
						 delayvec, smonvec, line,
						 inputs, &out_edge,
						 &outcycle, &out_comment) == 1)
				outeof = 1;
		} while (outeof == 0 && vecflag == 0);

		// EOF on either file?
		if (ineof || outeof)
			break;

		if(outcycle == -1) {
			// Backwards compatibility:
			// Rumage through the comment thats not really a comment and
			// Parse line number from .out file
			ptr = strchr(line, '#');
			if (ptr == NULL)
				outfile_error(1, "missing cycle number");
			ptr++;
			outcycle = atoi(ptr);
		}

		// Compare vectors from .in and .out file, write output, update rtnval, 
		if (process_vectors(cazmflag, hlflag, incycle, outcycle,
				    invec, outvec, sinvec, soutvec, 
				    (vecflag&4) ? smonvec : NULL,
				    in_edge, out_edge,
				    delayvec, inputs, dobase,
				    in_comment, out_comment))
			rtnval = 2;
	}

	if (ineof && !outeof)
		infile_error(1, "premature eof");

	if (!ineof && outeof)
		outfile_error(1, "premature eof");

	exit(rtnval);
}
