;
;;; spice-mode.el --- major mode for HSPICE netlist files, by Steve Tell
;; Based heavily on eldo-mode by Emmanuel Rouat
;;
;; Copyright (C) 1997,98 Emmanuel Rouat
;; Copyright (C) 1999 Steve Tell
;;
;; Author/maintainer: Steve Tell <tell@cs.unc.edu>
;; Created:         22 June 1999
;; based on eldo-mode 2 September 98 (0.92)
;; 
;; Version: 19990622
;; Keywords: HSPICE,spice,analog simulator,netlist editing mode
;;
;; TODO:
;; - more effective templates 
;;
;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.
;;
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
;;
;;; Commentary:
;;
;; This major mode for Xemacs provides support for
;; editing HSPICE netlist files.
;;
;; You may wish to add something like the following to your ~/.emacs file:
;; (autoload 'spice-mode "spice-mode" nil t)        
;; (setq auto-mode-alist (cons '("\\.sp$"  . spice-mode) auto-mode-alist))
;;
;; Put this file in the emacs load-path so emacs can find it; either in
;; the dedicated xemacs/site-lisp directory or in a private emacs lisp
;; directory.  To add a private emacs lisp directory to the load-path, 
;; add somthing like this to your .emacs:
;; (setq load-path (cons "/home/msl/tell/lib/elisp" load-path))
;;
;; This mode will activate automaticaly on files which end with .sp
;; This mode requires font-lock.
;;
;; Most of the code of this mode was shamelessly stolen from vhdl-mode (ron whitby)
;; and vhdl-electric mode - and from an emacs mode using 'hilit, written at Anacad
;; Also some borrowings from tcl-mode and zillions of .el files
;; Functions/subckt support from spice-mode by C.J Vieri
;; Good they were there... I HATE ELISP!!!!!


(require 'font-lock)

(defconst spice-mode-version "19990623"
  "Current version of spice mode.")

(defconst spice-developer "Steve Tell <tell@cs.unc.edu>"
  "Current developer/maintainer of spice-mode.")

(defvar spice-mode-hook nil
  "Function or functions to run on entry to spice-mode.")

(defvar spice-mode-menu
  '(["Comment Region"         comment-region (mark)]
    ["Uncomment Region"       uncomment-region (mark)]
    ["(Un)comment Region"     comment-uncomment-region (mark)]
    ["Fontify..."             font-lock-fontify-buffer t]
    ["Simulate"               compile  t]
    ["About Spice-Mode"        spice-about t])
  "XEmacs19 menu for Spice mode.")

; SPICE Electric Menus
(defvar spice-electric-mode-menu
  '(["Comment Region"         comment-region (mark)]
    ["Uncomment Region"       uncomment-region (mark)]
    ["(Un)comment Region"     comment-uncomment-region (mark)]
    ["Fontify..."             font-lock-fontify-buffer t]
    ["Simulate"               compile  t]
    "------"
;    ("Waveforms"
;     ["pulse"                  tempo-template-spice-pulse t]
;     ["ac"                     tempo-template-spice-ac t]
;     ["sin"                    tempo-template-spice-sine t]
;     ["exp"                    tempo-template-spice-exp t]
;     ["noise"                  tempo-template-spice-noise t]
;     ["pattern"                tempo-template-spice-pattern t]
;     )
    ("Elements"
;    ["Comp"                   tempo-template-spice-comp t]
;    ["Compd"                  tempo-template-spice-compd t]
;    ["Opamp0"                 tempo-template-spice-linear-opa t]
     ["nfet"                tempo-template-spice-nfet t]
     ["pfet"                tempo-template-spice-pfet t]
     )    "------"
    ("Commands"
     ["trans"                tempo-template-spice-trans t]
     )    "------"
    ["subckt"                 tempo-template-spice-subckt t]
    ["header"                 tempo-template-spice-circuit-header t]
    
    )
  "XEmacs19 popup menu for SPICE mode.")


(defvar spice-mode-syntax-table nil
  "The syntax table used in `spice-mode' buffers")

;; FIXME - doesn't work?
(defvar spice-end-comment-column (1- fill-column)
  "*Last column in Spice  comment (before line wraps)")

;; Define local faces
(copy-face 'font-lock-comment-face 'spice-comment-face)

;; Analysis cards (.op, tran etc)
(copy-face 'font-lock-keyword-face 'spice-analysis-face)

;; spicepp commmand keywords
(copy-face 'font-lock-type-face 'spice-command-face)

;; spice command keywords 
(copy-face 'font-lock-keyword-face 'spice-keyword-face)

;; circuit elements
(copy-face 'font-lock-variable-name-face 'spice-element-face)

;; In eldo you can also make a block comment, with
;; a #c (begin comment) and #e (end comment)


(if spice-mode-syntax-table
    ()
  (setq spice-mode-syntax-table (make-syntax-table (standard-syntax-table)))
    ;; XEmacs (formerly Lucid) has the best implementation
  (modify-syntax-entry ?#  ". 13" spice-mode-syntax-table)
  (modify-syntax-entry ?c  ". 2"   spice-mode-syntax-table)
  (modify-syntax-entry ?e  ". 4"   spice-mode-syntax-table)
  (modify-syntax-entry ?\n "> b"    spice-mode-syntax-table)
  (set-syntax-table spice-mode-syntax-table))

(defun spice-mode ()
  "Spice-mode is a major mode for highlighting HSPICE netlist files.
This mode requires font-lock.

You can use the 'compile' to start Spice with the current buffer as
input netlist.
Two types of comments are used:
- the '*' symbol is used to comment out a whole line - that symbol has
to be at the beginning of a line 
- the '$' symbol, at start of line or preceded by whitespace,
is used to document your netlist  

To add automatic support put something like the following in your .emacs file:
  \(autoload 'spice-mode \"spice-mode\" nil  t\)
  \(setq auto-mode-alist \(cons '\(\"\\\\.cir\" . spice-mode\) \
auto-mode-alist\)\)
  \(setq auto-mode-alist \(cons '\(\"\\\\.ckt\" . spice-mode\) \
auto-mode-alist\)\)  

spice-mode hacked up by Steve Tell, Based heavily on:
Eldo-mode Copyright (C) 1997,98 Emmanuel Rouat <emmanuel.rouat@wanadoo.fr>"

  (interactive)
  (setq tempo-interactive t)
  (setq mode-name "Spice")
  (setq major-mode 'spice-mode) 
  (set-syntax-table spice-mode-syntax-table)
  ;; put Spice menu into menubar for XEmacs
  (if (and (boundp 'current-menubar)
	   current-menubar
	   (not (assoc mode-name current-menubar)))
      (progn
	(set-buffer-menubar (copy-sequence current-menubar))
	(add-menu nil mode-name spice-mode-menu)))
  ;; put Spice electric menu into popup for XEmacs
  (if (boundp 'mode-popup-menu)
      (setq mode-popup-menu
	    (cons (concat mode-name " commands")
		  spice-electric-mode-menu)))


  (make-local-variable 'comment-start)
  (setq comment-start "*")  
  ;; use Spice as compiler, on current buffer
  (make-local-variable  'compile-command)
  (setq compile-command 
	(concat "hspice "
		(file-name-nondirectory buffer-file-name)
		))
  )

;; HSPICE is case-insensitive
(put 'spice-mode 'font-lock-keywords-case-fold-search t)

:; List of Spice keywords/syntax - order is important!!
(setq spice-mode-font-lock-keywords
 (list 
   ;; comments - lines starting with '*' are ignored by spice
   '("^\\*.*" 0 spice-comment-face t)

   ;; $ comments - start-of-line or whitespace followed by '$' ingnored to EOL
   '("[ \t]+\\$.*" 0 spice-comment-face t)
   '("^\\$.*" 0 spice-comment-face t)

   ;; analysis - keywords like .op, .tran etc are fontified as spice-analysis-face
;   '("^[ \t]*\\\.\\(ac\\|dc\\|op *$\\|mc\\|noisetran\\|noise\\|pz\\|sens\\|temp\\|tran\\|wcase\\|step\\)" . spice-analysis-face)

   ;; plot,probe,graph
 ;  '("^[ \t]*\\\.\\(plot\\|probe\\|graph\\)[ \t\n]+\\(ac\\|dc\\|tran\\|noise\\)" . spice-plot-face)

   ;; model ,subckt and ends statement
  ; '("^[ \t]*\\\.\\(model\\|subckt\\|ends\\)[ \t]+[^. \t\n]+" . spice-analysis-face)

   ;; other keywords - words beginning with . at start of line
   '("^\\.[^. \t*$]*" . spice-keyword-face)

   ;; spicepp extended commands - words beginning with @ at start of line
   '("^@[^. \t*$]*" . spice-command-face)

   ;; circuit elements keywords other words at start of line
   '("^[^. \n\t*$]+" . spice-element-face)
))


(defun uncomment-region (beg end)
  "Uncomment selected region - comment symbol is '*'
Doc comments (starting with '$') are unaffected."
  (interactive "r")
  (comment-region beg end -1))

(defun spice-comment-bar (&optional aligned)
  "Insert solid comment bar from column zero to end of line. If optional
   argument is provided, bar will be added from current column."
  (interactive)
  (if (not aligned) (beginning-of-line))
  (insert-char ?* (- spice-end-comment-column (current-column))))

(defun comment-uncomment-region (beg end &optional arg)
  "Comment out line in region if not commented out, uncomment it if already
commented out (exclusive-or type of comment) - comment symbol is '*' 
Doc comments (starting with '$') are unaffected"
  (interactive "r\nP")
  (goto-char beg)
  (narrow-to-region beg end)
  (while (not(eobp))
    (beginning-of-line)
    (if (looking-at comment-start)
     (delete-char (length comment-start))
     (if (looking-at "[ \t]*$") ()
              (insert comment-start) ) )
    (forward-line 1)
    )
  (widen)
  )

(defun spice-about ()
  (interactive)
  (sit-for 0)
  (message "Spice-mode version %s,  %s" spice-mode-version spice-developer))

;; Templates

(require 'tempo)

;; Waveforms

(tempo-define-template
 "spice-pulse"
 '("Pulse("
     (p "[start value]: ") " "  
     (p "[pulsed value]: ") " "
     (p "[delay]: ") " " 
     (p "[rise time]: ") " "
     (p "[fall time]: ") " " 
     (p "[pulse duration]: ") " " 
     (p "[period]: ")
     ")"'n) 
 "pulse"
 "template for inserting an SPICE Pulse waveform")

(tempo-define-template
 "spice-ac"
 '("ac("
     (p "[magnitude]: ") " "  
     (p "[phase]: ") " "
     ")"'n) 
 "ac"
 "template for inserting an SPICE AC waveform")

(tempo-define-template
 "spice-pattern"
 '("Pattern "
     (p "[Vhi]: ") " "  
     (p "[Vlo]: ") " "
     (p "[delay]: ") " " 
     (p "[rise time]: ") " "
     (p "[fall time]: ") " " 
     (p "[Bit duration]: ") " " 
     (p "[Bits]: ")
     'n) 
 "pattern"
 "template for inserting an SPICE Pattern function")

(tempo-define-template
 "spice-sine"
 '("sin("
     (p "[Offset]: ") " "  
     (p "[Amplitude]: ") " "
     (p "[Frequency]: ") " " 
     (p "[delay]: ") " " 
     (p "[Damping factor]: ") " " 
     ")"'n) 
 "sine"
 "template for inserting an SPICE SINE function")

(tempo-define-template
 "spice-exp"
 '("exp("
     (p "[start value]: ") " "  
     (p "[target value]: ") " "
     (p "[rise delay]: ") " "
     (p "[tau1]: ") " "
     (p "[fall delay]: ") " "
     (p "[tau2]: ") " "
     ")"'n) 
 "exp"
 "template for inserting an SPICE EXP waveform")

(tempo-define-template
 "spice-noise"
 '("noise("
     (p "[White noise level]: ") " "  
     (p "[Flicker noise level]: ") " "
     (p "[Alpha]: ") " "
     (p "[Cut-off freq]: ") " "
     (p "[Filter order]: ") " "
     ")"'n) 
 "noise"
 "template for inserting an SPICE NOISE waveform")

;; circuit elements

(tempo-define-template
 "spice-nfet"
 '('& "m"
     (p "[Instance name]: ") " "  
     (p "[Source]: ") " "
     (p "[Gate]: ") " "
     (p "[Drain]: ") " "
     (p "[Substrate]: ")
     " nfet l=" (p "[Length]: ")
     " w=" (p "[Width]: ") " "
     'n) 
 "linear opad"
 "template for inserting an SPICE nmos fet")

(tempo-define-template
 "spice-pfet"
 '('& "m"
     (p "[Instance name]: ") " "  
     (p "[Source]: ") " "
     (p "[Gate]: ") " "
     (p "[Drain]: ") " "
     (p "[Substrate]: ")
     " pfet l=" (p "[Length]: ")
     " w=" (p "[Width]: ") " "
     'n) 
 "linear opad"
 "template for inserting an SPICE pmos fet")

;; control cards

(tempo-define-template
 "spice-trans"
 '('& ".trans "
     (p "[Increment]: ") " "  
     (p "[Duration]: ")
     'n) 
 "trans"
 "template for inserting a SPICE .transient statement")

;; SUBCKT and HEADER

(tempo-define-template
 "spice-subckt"
 '(".SUBCKT "
   (p "[subckt name]: " lname) 'r 'n 'n
   ".ENDS " (s lname)  '>)
 "subckt"
 "template for inserting an Spice subckt")

(tempo-define-template
 "spice-circuit-header"
 '((spice-comment-bar) 'n
   "* "'n
   "*-- Circuit name  : " (buffer-name) 'n
   "*-- Designer(s)   : " 'r 'n 
   "*-- Library       : " 'n
   "*-- Purpose       : " 'n
   "*-- Inputs        : " 'n
   "*-- Outputs       : " 'n
   "*-- Supplies      : " 'n
   "*-- REFERENCES    : " 'n
   "*-- "'n
   (spice-comment-bar) 'n)
 "header"
 "template for inserting a header for a circuit")

(provide 'spice-mode)

;;; spice-mode.el ends here
