/* parse.y --- yacc-code for tcalc parser

MODULE
    parse.y

DESCRIPTION
    This is the YACC code which describes the grammar of input language
    for the tcalc analyzer.  
    
    When YACC is run on this file, it must be done with the -d option.
    This causes it to write out the header file y.tab.h, which includes
    the token definitions needed for LEX as well as the union define
    for YYSTYPE (If you don't understand this, see the YACC manual).

HISTORY
    6/6/88  --- written by Steven Molnar
    2/14/91 --- modified to swallow new CAzM header by Steve Molnar
    2-Nov-98 --- accept '-' to read stdin, use getopt(3) Steve Tell

BUGS
    Only reads cazm format input file

---------------------------------------------------------*/

%{

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "tcalc.h"

extern FILE *yyin;




/***********************************************************************/
/* Global variable declarations                                        */

float Start = 0.0;		  /* time start value */
float Vdd   = DEFAULT_VDD;	  /* global threshold parameters */
float L     = DEFAULT_VDD*0.1;
float X     = DEFAULT_VDD*0.5;
float H     = DEFAULT_VDD*0.9;

SIGNAL *SignalList[MAX_SIGNALS];  /* table of defined electrical signals */
int     NumSignals = 0;
SIGNAL *TraceList[MAX_SIGNALS];   /* table of signals being traced */
int     NumTraces = 0;

VAR *VarList[MAX_VARS];	          /* time variables */
int  NumVars = 0;

float Time;			  /* current time value */
float OldTime;			  /* time value at last tick */

int   ExitCode = 0;		  /* exit value of program */

/***********************************************************************/
/* Forward and external procedure declarations                         */

VAR *new_var();
VAR *lookup_var();
extern float expr_eval();
void barf();

%}

%union {
  int     s_int;
  float   s_float;
  char    s_char[MAX_NAME_LEN];
  EXPR   *s_expr;
  SIGNAL *s_signal;
  VAR    *s_var;
  }

%token EOLN LOW CROSS HIGH VDD RISE FALL POWER VOLTAGE
%token START TIME COMMA COLON LPAREN RPAREN EQUAL CAZM_HEADER
%left  MIN MAX
%left  PLUS MINUS
%left  MULT DIVIDE
%left  UMINUS
%token <s_float>  FLOAT
%token <s_char>   IDENT 
%type  <s_expr>   expr 
%type  <s_signal> signal
%type  <s_var>    var  
%type  <s_int>    rf
%type  <s_int>    transition
%type  <s_int>    paramlist

%%

process		: decls traces
		;

decls           : /* empty */
                | decls decl EOLN
                ;

decl            : /* empty */
                | error 
                | START EQUAL FLOAT
                    {
		      Start = $3;
		    }
                | VDD EQUAL FLOAT
                    {
		      Vdd = $3;
		      L   = 0.1 * $3;
		      X   = 0.5 * $3;
		      H   = 0.9 * $3;
		    }
                | LOW EQUAL FLOAT
                    {
		      L = $3;
		    }
                | CROSS EQUAL FLOAT
                    {
		      X = $3;
		    }
                | HIGH EQUAL FLOAT
                    {
		      H = $3;
		    }
                | VDD signal EQUAL FLOAT
                    {
		      $2->Vdd = $4;
		      $2->L   = 0.1 * $4;
		      $2->X   = 0.5 * $4;
		      $2->H   = 0.9 * $4;
		    }
                | LOW signal EQUAL FLOAT
                    {
		      $2->L   = $4;
		    }
                | CROSS signal EQUAL FLOAT
                    {
		      $2->X = $4;
		    }
                | HIGH signal EQUAL FLOAT
                    {
		      $2->H = $4;
		    }
                | var EQUAL expr
                    {
		      $1->expr = $3;
		    }
                ;
  

/* Rules to handle expressions */

expr            : LPAREN expr RPAREN
                    {
		      $$ = $2;
		    }
                | expr MIN expr
                    {
		      $$ = (EXPR *) malloc(sizeof(EXPR));
		      $$->type = MIN;
		      $$->arg1 = $1;
		      $$->arg2 = $3;
		    }
                | expr MAX expr
                    {
		      $$ = (EXPR *) malloc(sizeof(EXPR));
		      $$->type = MAX;
		      $$->arg1 = $1;
		      $$->arg2 = $3;
		    }
                | expr PLUS expr
                    {
		      $$ = (EXPR *) malloc(sizeof(EXPR));
		      $$->type = PLUS;
		      $$->arg1 = $1;
		      $$->arg2 = $3;
		    }
                | expr MINUS expr
                    {
		      $$ = (EXPR *) malloc(sizeof(EXPR));
		      $$->type = MINUS;
		      $$->arg1 = $1;
		      $$->arg2 = $3;
		    }
                | expr MULT expr
                    {
		      $$ = (EXPR *) malloc(sizeof(EXPR));
		      $$->type = MULT;
		      $$->arg1 = $1;
		      $$->arg2 = $3;
		    }
                | expr DIVIDE expr
                    {
		      $$ = (EXPR *) malloc(sizeof(EXPR));
		      $$->type = DIVIDE;
		      $$->arg1 = $1;
		      $$->arg2 = $3;
		    }
                | MINUS expr   %prec UMINUS
                    {
		      $$ = (EXPR *) malloc(sizeof(EXPR));
		      $$->type  = UMINUS;
		      $$->arg1 = (EXPR *) malloc(sizeof(EXPR));
		      $$->arg1->type  = FLOAT;
		      $$->arg1->contents.value = 0.0;
		      $$->arg2 = $2;
		    }
                | var
                    {
		      $$ = (EXPR *) malloc(sizeof(EXPR));
		      $$->type = IDENT;
		      $$->contents.var  = $1;
		    }
                | signal COLON rf COMMA transition COMMA FLOAT
                    {
		      $$ = (EXPR *) malloc(sizeof(EXPR));
		      $$->type = CROSS;
		      $$->contents.trans.signal = $1;
		      $$->contents.trans.rf     = $3;
		      $$->contents.trans.trans  = $5;
		      $$->contents.trans.num    = (int)$7;
		      if ($7 != (float)(int)$7 || $7 < 0.0)
			error_out("Cycle number must be integer >= 0");
		    }
                | FLOAT
                    {
		      $$ = (EXPR *) malloc(sizeof(EXPR));
		      $$->type = FLOAT;
		      $$->contents.value = $1;
		    }
                ;

/* Enumerate rise/fall and transition tokens */

rf              : RISE
                    {
		      $$ = RISE;
		    }
                | FALL
                    {
		      $$ = FALL;
		    }
                ;

transition      : LOW
                    {
		      $$ = LOW;
		    }
                | CROSS
                    {
		      $$ = CROSS;
		    }
                | HIGH
                    {
		      $$ = HIGH;
		    }
                ;

/* Rules for trace section */

traces          : /* empty */
                | header TIME vars EOLN data 
                | header TIME vars EOLN data POWER /* ignore power info */
                ;

header          : CAZM_HEADER EOLN EOLN IDENT IDENT EOLN
                    {
		      if (strcmp($4, "TRANSIENT") != 0)
			error_out("garbled CAzM file header\n");
		    }
                ;

vars            : /* nothing */
                | vars VOLTAGE LPAREN signal RPAREN
                    {
		      TraceList[NumTraces++] = $4;
		    }
                | vars signal
                    {
		      TraceList[NumTraces++] = $2;
		    }
                ;

/* Rules for parsing lines of trace data */

data            : /* nothing */
                | data paramlist EOLN
  		    {
		      if ($2 == NumTraces)
			{
			  tick();
			  OldTime = Time;
			}
		      else if ($2 != -1) /* not empty line */
			error_out("bad data format");
		    }
                ;

paramlist       : /* empty */
		    {
		      $$ = -1;
		    }
                | paramlist FLOAT
		    {
		      if ($1 == -1)
			Time = $2;
		      else
			TraceList[$1]->V = $2;
		      $$ = $1+1;
		    }
                ;

signal            : IDENT
                    {
		      SIGNAL *temp;

		      temp = lookup_signal($1);
		      if (temp)
			$$ = temp;
		      else
			$$ = new_signal($1);
		    }
                ;


var             : IDENT
                    {
		      VAR *temp;

		      temp = lookup_var($1);
		      if (temp)
			$$ = temp;
		      else
			$$ = new_var($1);
		    }
                ;

%%

/*************************************************************************
**************************************************************************
**     Support routines for lexical analyzer and parser                 **
**************************************************************************
*************************************************************************/


/************************************************************************
Create a signal instance.  Initialize it.  Add it to SignalList.

On entry:
  name = name of signal
************************************************************************/
SIGNAL *new_signal (name)
     char *name;
{
  SIGNAL *new;

  new = (SIGNAL *) malloc (sizeof(SIGNAL));
  strcpy(new->name, name);
  new->Vdd   = Vdd;
  new->L     = L;
  new->X     = X;
  new->H     = H;
  new->state = PRE;
  new->nRL   = new->nRX = new->nRH =
  new->nFL   = new->nFX = new->nFH = 0;
  if (NumSignals < MAX_SIGNALS)
    SignalList[NumSignals++] = new;
  else
    barf("too many signals");
  return(new);
}


/************************************************************************
Create a time variable instance.  Initialize it.  Add it to VarList.

On entry:
  name = name of time variable
************************************************************************/
VAR *new_var (name)
     char *name;
{
  VAR *new;

  new = (VAR *) malloc (sizeof(VAR));
  strcpy(new->name, name);
  if (NumVars < MAX_VARS)
    VarList[NumVars++] = new;
  else
    barf("too many variables");
  return(new);
}


/************************************************************************
Return pointer to signal specified by 'name' (NULL if not found).

On entry:
  name = name of signal
On exit:
  return pointer to signal (NULL if not found)
************************************************************************/
SIGNAL *lookup_signal (name)
     char *name;
{
  int i;

  for (i = 0; i < NumSignals; i++)
    if (strcmp(name, SignalList[i]->name) == 0)
      return(SignalList[i]);
  
  return((SIGNAL *) NULL);
}


/************************************************************************
Return pointer to variable specified by 'name' (NULL if not found).

On entry:
  name = name of variable
On exit:
  return pointer to variable (NULL if not found)
************************************************************************/
VAR *lookup_var (name)
     char *name;
{
  int i;

  for (i = 0; i < NumVars; i++)
    if (strcmp(name, VarList[i]->name) == 0)
      return(VarList[i]);
  
  return((VAR *) NULL);
}


/************************************************************************
Make sure that all defined signals appear in TraceList.
************************************************************************/
void check_signals()
{
  int i, j, okay;

  for (i = 0; i < NumSignals; i++)
    {
      okay = FALSE;

      for (j = 0; j < NumTraces; j++)
	if (SignalList[i] == TraceList[j])
	  okay = TRUE;

      if (!okay)
	fprintf(stderr,"Warning: signal <%s> defined but not used\n",
		SignalList[i]->name);
    }
}


/************************************************************************
Evaluate the expressions defining each variable.
************************************************************************/
void
eval_vars()
{
  int i;

  for (i = 0; i < NumVars; i++)
    if (VarList[i]->expr)
      {
	VarList[i]->value = expr_eval(VarList[i]->expr);
	VarList[i]->valid = TRUE;
	printf("%s = %g\n",VarList[i]->name, VarList[i]->value);
      }
}


/************************************************************************
Syntax error routine.  Called when YACC perceives a syntax error.
************************************************************************/
void
yyerror (s)
     char *s;
{
  error_out(s);
}


/************************************************************************
Barf and die.  Print out last words.
************************************************************************/
void
barf (s)
     char *s;
{
  fprintf(stderr,"Fatal error: %s\n",s);
  exit(1);
}


/*************************************************************************
Prints a message informing the correct usage of the command, then exits.
****************************************************************************/
void
usage()
{
  fprintf(stderr, "\nUsage: tcalc [-V] [-v Vdd] [-s tzero] file*\n\n");
  exit(1);
}


/*************************************************************************
Main routine for tcalc. 
****************************************************************************/
int
main(int argc, char **argv)	/* main routine */
{
 	extern int optind;
	extern char *optarg;
	int c;

	int   verbose = FALSE;

	while ((c = getopt (argc, argv, "Vv:s:")) != EOF) {
		switch(c) {
		case 'V':
			verbose = 1;
			break;
		case 'v':
			Vdd = atof(optarg);
			H = 0.9*Vdd;
			X = 0.5*Vdd;
			L = 0.1*Vdd;
			break;
			
		case 's':
			Start = atof(optarg);
			break;

		default:
			ExitCode = 1;
			break;
		}
	}

	if(ExitCode) {
		usage();
		exit(ExitCode);
	}

  /* Process input */
	if(optind >= argc) { /* no files, use stdin */
		init_lexer("<stdin>");
		if(verbose)
			fprintf(stderr, "reading stdin\n");
		yyparse();
	} else {
		for(; optind < argc; optind++) { /* files or stdin */
			if(strcmp("-", argv[optind]) == 0) {
				init_lexer("<stdin>");
				yyin = stdin;
				if(verbose)
					fprintf(stderr, "reading stdin\n");
				yyparse();
			} else {
				init_lexer(argv[optind]);
				if ((yyin=fopen(argv[optind],"r"))==NULL) {
					perror(argv[optind]);
					exit(1);
				} else {
					if(verbose)
						fprintf(stderr, "reading %s\n", argv[optind]);
					yyparse();
					/* fclose(yyin); */
				}
			}
		}
	}
	check_signals();

	if (verbose)
		print_transitions();

	eval_vars();

	exit(ExitCode);
}
